/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: PDChannelLogger_CCA.h                                     //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2024 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////
#ifndef __PDChannelLogger_CCA_H__
#define __PDChannelLogger_CCA_H__

#include "CCA_Core.h"


#ifdef CCA_PACK
    #pragma pack(push, CCA_PACK)
#endif

/** BrainStem Power Delivery Packet Structure - Contains information representing a
 * Power Delivery packet along with contextual device information */
struct BS_PD_Packet_CCA {
    unsigned char channel;      /**< Channel/Index */
    unsigned int seconds;       /**< Seconds in device time since power on. */
    unsigned int uSeconds;      /**< Micro Seconds in device time since power on .*/
    unsigned char direction;    /**< Direction of packet transmission relative to the device. */
    unsigned char sop;          /**< See bs_pd_packet.h for more details */
    unsigned int event;         /**< Packet type - See powerdeliveryLogEvent in aProtocolDefs.h */
    unsigned int payloadSize;   /**< Length of the payload. */
    unsigned char* payload;     /**< Raw PD Packet data */
};

#ifdef CCA_PACK
    #pragma pack(pop)
#endif


/// \defgroup PDChannelLogger PD Channel Logger
/// PDChannelLogger:
/// Provides an interface for managing BrainStem Power Delivery Packets.
/// Packets are accepted and decoded asynchronously.
/// Pay careful attention to packet cleanup. Portions of the structure do not belong
/// to the caller.
///

#ifdef __cplusplus
extern "C" {
#endif

    /// Creates internal object for managing BrainStem Power Delivery logging packets.
    /// \param id ID assigned through "module_createStem"
    /// \param result Object containing aErrNone and the requested value on success.
    /// Non-zero error code on failure.
    ///    The current upstream port.
    /// \param index The index of the entity in question.
    /// \param bufferLength Number of packets the class should queue before dropping.
    /// \details Returns \ref EntityReturnValues "common entity" return values
    aLIBEXPORT void __stdcall PDChannelLogger_create(unsigned int* id, struct Result* result, const int index, const unsigned int bufferLength);

    /// Destroys internal object for managing BrainStem Power Delivery logging packets.
    /// \param id ID assigned through "module_createStem"
    /// \param result Object containing aErrNone and the requested value on success.
    /// Non-zero error code on failure.
    ///    The current upstream port.
    /// \param index The index of the entity in question.
    /// \details Returns \ref EntityReturnValues "common entity" return values
    aLIBEXPORT void __stdcall PDChannelLogger_destroy(unsigned int* id, struct Result* result, const int index);

    /// Enables Power Delivery logging.
    /// \param id ID assigned through "module_createStem"
    /// \param result Object containing aErrNone and the requested value on success.
    /// Non-zero error code on failure.
    ///    The current upstream port.
    /// \param index The index of the entity in question.
    /// \param enable True enables logging; False disables logging
    /// \details True on success.
    aLIBEXPORT void __stdcall PDChannelLogger_setEnabled(unsigned int* id, struct Result* result, const int index, const bool enable);

    /// Attempts to takes a packet from the internal buffer.
    /// \param id ID assigned through "module_createStem"
    /// \param result Object containing aErrNone and the requested value on success.
    /// Non-zero error code on failure.
    ///    The current upstream port.
    /// \param index The index of the entity in question.
    /// \param packet Reference to a packet to be filled by the function.
    /// \details True if the function successfully acquired any number of packets. False if no packets were available.
    aLIBEXPORT void __stdcall PDChannelLogger_getPacket(unsigned int* id, struct Result* result, const int index, struct BS_PD_Packet_CCA* packet);

    /// Attempts to take a multiple packets (up to a maximum) from the internal buffer.
    /// \param id ID assigned through "module_createStem"
    /// \param result Object containing aErrNone and the requested value on success.
    /// Non-zero error code on failure.
    ///    The current upstream port.
    /// \param index The index of the entity in question.
    /// \param packetBuffer pointer to a buffer to be filled
    /// \param bufferLength The length of the buffer provided.
    /// \details True if the function successfully acquired any number of packets. False if no packets were available.
    aLIBEXPORT void __stdcall PDChannelLogger_getPackets(unsigned int* id, struct Result* result, const int index, struct BS_PD_Packet_CCA* packetBuffer, const unsigned int bufferLength);

    /// Releases the internal memory contained within the payload parameter.
    /// \param id ID assigned through "module_createStem"
    /// \param result Object containing aErrNone and the requested value on success.
    /// Non-zero error code on failure.
    ///    The current upstream port.
    /// \param packet Reference to a packet to have its memory freed.
    /// \details Returns \ref EntityReturnValues "common entity" return values
    aLIBEXPORT void __stdcall PDChannelLogger_freePayloadBuffer(unsigned int* id, struct Result* result, struct BS_PD_Packet_CCA* packet);

#ifdef __cplusplus
}
#endif

#endif
