/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: PDChannelLogger.hpp                                       //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Description: Class for managing PD streaming packets            //
// Automatically queues BrainStem Power Delivery packets.          //
//                                                                 //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2022 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __PDChannelLogger_H__
#define __PDChannelLogger_H__


#include <stdint.h>

#include "bs_pd_packet.h"
#include "BrainStem-core.h"

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {
namespace Utilities {

class PDChannelLogger_Private;

class aLIBEXPORT PDChannelLogger {
public:
    
    /// Manages BrainStem Power Delivery logging packets.
    /// \param module Reference to an existing BrainStem Module
    /// \param index Index/channel logging should be enabled for.
    /// \param bufferLength Number of packets the class should queue before dropping.
    PDChannelLogger(Acroname::BrainStem::Module* module, const uint8_t index, const uint16_t bufferLength = 1024);
    ~PDChannelLogger();
    
    /// Enables Power Delivery logging.
    /// \param enable True enables logging; False disables logging
    /// \return aErrNone on sucess
    aErr setEnabled(bool enable);
    
    /// Attempts to takes a packet from the internal buffer.
    /// \param packet Reference to a packet to be filled by the function.
    /// \return aErrNone if the function successfully acquired any number of packets.\n
    ///  aErrNotReady if no packets were available.\n
    ///  aErrInitialization if this module is not in a valid state to execute.
    aErr getPacket(BS_PD_Packet_t& packet);
    
    /// Attempts to take a multiple packets (up to a maximum) from the internal buffer.
    /// \param packets reference to a vector to be filled by the function
    /// \param maxPackets The maximum number of packets to acquire.
    /// \return aErrNone if the function successfully acquired any number of packets.\n
    ///  aErrNotReady if no packets were available.\n
    ///  aErrInitialization if this module is not in a valid state to execute.
    aErr getPackets(std::vector<BS_PD_Packet_t>& packets, const unsigned int maxPackets = 200);
    
    /// Provides the index for which this object was created for.
    /// \return The index of the object.
    uint8_t getIndex(void) const { return _index; }
    
private:
    Acroname::BrainStem::Module* _module;
    const uint8_t _index;
    PDChannelLogger_Private* _impl;
};

} //End namespace Utilities
} //End namespace BrainStem
} //End namespace Acroname

#endif//defined(__cplusplus)

#endif /* __PDChannelLogger_H__ */
