/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_AnalogClass_CPP.h                                 //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2025 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_ANALOG_CPP_H__
#define __AUTOGEN_ANALOG_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// AnalogClass:
/// Interface to analog entities on BrainStem modules.
/// Analog entities may be configured as a input or output depending on hardware capabilities.
/// Some modules are capable of providing actual voltage readings, while other simply return
///   the raw analog-to-digital converter (ADC) output value.
/// The resolution of the voltage or number of useful bits is also hardware dependent.
/// 
class aLIBEXPORT AnalogClass : public EntityClass {

public:

    /// Constructor.
    AnalogClass(void);

    /// Destructor.
    virtual ~AnalogClass(void);

    /// Initialize the Analog Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the Analog entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Get the raw ADC output value in bits.
    /// 
    /// \param value 16 bit analog reading with 0 corresponding to the negative analog voltage
    ///        reference and 0xFFFF corresponding to the positive analog voltage reference.
    /// 
    /// \note Not all modules are provide 16 useful bits; this value's least significant bits are
    ///       zero-padded to 16 bits.
    ///       Refer to the module's datasheet to determine analog bit depth and reference voltage.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getValue(uint16_t* value);

    /// Get the scaled micro volt value with reference to ground.
    /// 
    /// \param microvolts 32 bit signed integer (in microvolts) based on the board's ground and
    ///        reference voltages.
    /// 
    /// \note Not all modules provide 32 bits of accuracy.
    ///       Refer to the module's datasheet to determine the analog bit depth and reference
    ///       voltage.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVoltage(int32_t* microvolts);

    /// Get the analog input range.
    /// 
    /// \param range 8 bit value corresponding to a discrete range option
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getRange(uint8_t* range);

    /// Get the analog output enable status.
    /// 
    /// \param enable 0 if disabled 1 if enabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getEnable(uint8_t* enable);

    /// Set the value of an analog output (DAC) in bits.
    /// 
    /// \param value 16 bit analog set point with 0 corresponding to the negative analog voltage
    ///        reference and 0xFFFF corresponding to the positive analog voltage reference.
    /// 
    /// \note Not all modules are provide 16 useful bits; the least significant bits are
    ///       discarded.
    ///       E.g. for a 10 bit DAC, 0xFFC0 to 0x0040 is the useful range.
    ///       Refer to the module's datasheet to determine analog bit depth and reference voltage.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setValue(const uint16_t value);

    /// Set the voltage level of an analog output (DAC) in microvolts.
    /// 
    /// \param microvolts 32 bit signed integer (in microvolts) based on the board's ground and
    ///        reference voltages.
    /// 
    /// \note Voltage range is dependent on the specific DAC channel range.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setVoltage(const int32_t microvolts);

    /// Set the analog input range.
    /// 
    /// \param range 8 bit value corresponding to a discrete range option
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setRange(const uint8_t range);

    /// Set the analog output enable state.
    /// 
    /// \param enable set 1 to enable or 0 to disable.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setEnable(const uint8_t enable);

    /// Set the analog configuration.
    /// 
    /// \param configuration bitAnalogConfigurationOutput configures the analog entity as an
    ///        output.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setConfiguration(const uint8_t configuration);

    /// Get the analog configuration.
    /// 
    /// \param configuration Current configuration of the analog entity.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getConfiguration(uint8_t* configuration);

    /// Set the sample rate for this analog when bulk capturing.
    /// 
    /// \param value sample rate in samples per second (Hertz).
    ///        - Minimum rate: 7,000 Hz
    ///        - Maximum rate: 200,000 Hz
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setBulkCaptureSampleRate(const uint32_t value);

    /// Get the current sample rate setting for this analog when bulk capturing.
    /// 
    /// \param value upon success filled with current sample rate in samples per second (Hertz).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getBulkCaptureSampleRate(uint32_t* value);

    /// Set the number of samples to capture for this analog when bulk capturing.
    /// 
    /// \param value number of samples.
    ///        - Minimum # of Samples: 0
    ///        - Maximum # of Samples: (BRAINSTEM_RAM_SLOT_SIZE / 2) = (3FFF / 2) = 1FFF = 8191
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setBulkCaptureNumberOfSamples(const uint32_t value);

    /// Get the current number of samples setting for this analog when bulk capturing.
    /// 
    /// \param value number of samples.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getBulkCaptureNumberOfSamples(uint32_t* value);

    /// Initiate a BulkCapture on this analog.
    /// Captured measurements are stored in the module's RAM store (RAM_STORE) slot 0.
    /// Data is stored in a contiguous byte array with each sample stored in two consecutive
    ///   bytes, LSB first.
    /// 
    /// \note When the bulk capture is complete getBulkCaptureState() will return either
    ///       bulkCaptureFinished or bulkCaptureError.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr initiateBulkCapture(void);

    /// Get the current bulk capture state for this analog.
    /// 
    /// \param state the state of bulk capture.
    ///        - Idle: bulkCaptureIdle = 0
    ///        - Pending: bulkCapturePending = 1
    ///        - Finished: bulkCaptureFinished = 2
    ///        - Error: bulkCaptureError = 3
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getBulkCaptureState(uint8_t* state);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_ANALOG_CPP_H__