/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_DigitalClass_CPP.h                                //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2025 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_DIGITAL_CPP_H__
#define __AUTOGEN_DIGITAL_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// DigitalClass:
/// Interface to digital entities on BrainStem modules.
/// Digital entities have the following 5 possibilities: Digital Input, Digital Output,
///   RCServo Input, RCServo Output, and HighZ.
/// Other capabilities may be available and not all pins support all configurations. Please
///   see the product datasheet.
/// 
class aLIBEXPORT DigitalClass : public EntityClass {

public:

    /// Constructor.
    DigitalClass(void);

    /// Destructor.
    virtual ~DigitalClass(void);

    /// Initialize the Digital Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the Digital entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Set the digital configuration to one of the available 5 states.
    /// 
    /// \param configuration The configuration to be applied
    ///            - Digital Input: digitalConfigurationInput = 0
    ///            - Digital Output: digitalConfigurationOutput = 1
    ///            - RCServo Input: digitalConfigurationRCServoInput = 2
    ///            - RCServo Output: digitalConfigurationRCServoOutput = 3
    ///            - High Z State: digitalConfigurationHiZ = 4
    ///            - Digital Input: digitalConfigurationInputPullUp = 0
    ///            - Digital Input: digitalConfigurationInputNoPull = 4
    ///            - Digital Input: digitalConfigurationInputPullDown = 5
    /// 
    /// \note Some configurations are only supported on specific pins.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setConfiguration(const uint8_t configuration);

    /// Get the digital configuration.
    /// 
    /// \param configuration Current configuration of the digital entity.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getConfiguration(uint8_t* configuration);

    /// Set the logical state.
    /// 
    /// \param state The state to be set. 0 is logic low, 1 is logic high.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setState(const uint8_t state);

    /// Get the state.
    /// 
    /// \param state The current state of the digital entity. 0 is logic low, 1 is logic high.
    /// 
    /// \note If in high Z state an error will be returned.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getState(uint8_t* state);

    /// Sets the logical state of all available digitals based on the bit mapping.
    /// Number of digitals varies across BrainStem modules.
    /// Refer to the datasheet for the capabilities of your module.
    /// 
    /// \param state The state to be set for all digitals in a bit mapped representation.
    ///        0 is logic low, 1 is logic high. Where bit 0 = digital 0, bit 1 = digital 1 etc.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setStateAll(const uint32_t state);

    /// Gets the logical state of all available digitals in a bit mapped representation.
    /// Number of digitals varies across BrainStem modules.
    /// Refer to the datasheet for the capabilities of your module.
    /// 
    /// \param state The state of all digitals where bit 0 = digital 0,
    ///        bit 1 = digital 1 etc. 0 is logic low, 1 is logic high.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getStateAll(uint32_t* state);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_DIGITAL_CPP_H__