/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_HDBaseTClass_CPP.h                                //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2025 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_HDBASET_CPP_H__
#define __AUTOGEN_HDBASET_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// HDBaseTClass:
/// This entity is only available on certain modules, and provides information on HDBaseT
///   extenders.
/// 
class aLIBEXPORT HDBaseTClass : public EntityClass {

public:

    /// Constructor.
    HDBaseTClass(void);

    /// Destructor.
    virtual ~HDBaseTClass(void);

    /// Initialize the HDBaseT Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the HDBaseT entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Gets the serial number of the HDBaseT device (6 bytes)
    /// 
    /// \param buffer pointer to the start of a c style buffer to be filled
    /// \param bufferLength Length of the buffer to be filed
    /// \param unloadedLength Length that was actually received and filled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSerialNumber(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Gets the firmware version of the HDBaseT device
    /// 
    /// \param firmwareVersion A bit packet representation of the firmware version
    ///        Major: Bits 24-31; Minor: Bits 16-23; Patch: Bits 8-15; Build: Bits 0-7
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getFirmwareVersion(uint32_t* firmwareVersion);

    /// Gets the current state of the HDBaseT link
    /// 
    /// \param state Bit packeted representation of the state.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getState(uint32_t* state);

    /// Gets the perceived cable length
    /// 
    /// \param cableLength Cable length in micro-meters
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCableLength(uint32_t* cableLength);

    /// Gets the Mean Squared Error (MSE) for channel A
    /// 
    /// \param mseA The current MSE for channel A in micro-dB
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getMSEA(int32_t* mseA);

    /// Gets the Mean Squared Error (MSE) for channel B
    /// 
    /// \param mseB The current MSE for channel B in micro-dB
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getMSEB(int32_t* mseB);

    /// Gets the number of successful messages between retransmission
    /// 
    /// \param retransmissionRate Instantaneous number of successful messages between
    ///        retransmission.
    ///        To be interpreted as: 1 / retransmissionRate for rate interpretation.
    ///        If the value is 0, there have been no retransmissions, otherwise higher is better..
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getRetransmissionRate(uint32_t* retransmissionRate);

    /// Gets the current link utilization
    /// 
    /// \param linkUtilization Utilization in milli-percent
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getLinkUtilization(uint32_t* linkUtilization);

    /// Gets the current encoding state.
    /// 
    /// \param encodingState Signal modulation encoding type.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getEncodingState(uint8_t* encodingState);

    /// Gets the USB2 tree at the HDBaseT device.
    /// 
    /// \param buffer pointer to the start of a c style buffer to be filled
    /// \param bufferLength Length of the buffer to be filed
    /// \param unloadedLength Length that was actually received and filled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getUSB2DeviceTree(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Gets the USB3 tree at the HDBaseT device.
    /// 
    /// \param buffer pointer to the start of a c style buffer to be filled
    /// \param bufferLength Length of the buffer to be filed
    /// \param unloadedLength Length that was actually received and filled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getUSB3DeviceTree(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Gets the current link role
    /// In the case of "Auto" the getState API will provide the current role.
    /// 
    /// \param role Link role
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getLinkRole(uint8_t* role);

    /// Sets the active link role
    /// 
    /// \param role The role to be set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setLinkRole(const uint8_t role);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_HDBASET_CPP_H__