/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_I2CClass_CPP.h                                    //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2025 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_I2C_CPP_H__
#define __AUTOGEN_I2C_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// I2CClass:
/// Interface the I2C buses on BrainStem modules.
/// The class provides a way to send read and write commands to I2C devices on the entities
///   bus.
/// 
class aLIBEXPORT I2CClass : public EntityClass {

public:

    /// Constructor.
    I2CClass(void);

    /// Destructor.
    virtual ~I2CClass(void);

    /// Initialize the I2C Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the I2C entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Read from a device on this I2C bus.
    /// 
    /// \param address The I2C address (7bit <XXXX-XXX0>) of the device to read.
    /// \param readLength The length of the data to read in bytes.
    /// \param buffer The array of bytes that will be filled with the result, upon success.
    ///        This array should be larger or equivalent to aBRAINSTEM_MAXPACKETBYTES - 5
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr read(const uint8_t address, const uint8_t readLength, uint8_t* buffer);

    /// Write to a device on this I2C bus.
    /// 
    /// \param address The I2C address (7bit <XXXX-XXX0>) of the device to write.
    /// \param bufferLength The length of the data to write in bytes.
    /// \param buffer The data to send to the device
    ///        This array should be no larger than aBRAINSTEM_MAXPACKETBYTES - 5
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr write(const uint8_t address, const uint8_t bufferLength, const uint8_t* buffer);

    /// Set bus pull-up state.
    /// This call only works with stems that have software controlled pull-ups.
    /// Check the datasheet for more information.
    /// This parameter is saved when system.save is called.
    /// 
    /// \param enable true enables pull-ups false disables them.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPullup(const uint8_t enable);

    /// Set I2C bus speed.
    /// 
    /// This call sets the communication speed for I2C transactions through this API.
    /// Speed is an enumeration value which can take the following values:
    ///     1 - 100Khz
    ///     2 - 400Khz
    ///     3 - 1MHz
    /// 
    /// \param speed The speed setting value.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setSpeed(const uint8_t speed);

    /// Get I2C bus speed.
    /// 
    /// This call gets the communication speed for I2C transactions through this API.
    /// Speed is an enumeration value which can take the following values:
    ///     1 - 100Khz
    ///     2 - 400Khz
    ///     3 - 1MHz
    /// 
    /// \param speed The speed setting value.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSpeed(uint8_t* speed);


private:
    uint8_t _busSpeed = i2cDefaultSpeed;


};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_I2C_CPP_H__