/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_MuxClass_CPP.h                                    //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2025 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_MUX_CPP_H__
#define __AUTOGEN_MUX_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// MuxClass:
/// A MUX is a multiplexer that takes one or more similar inputs (bus, connection, or signal)
///   and allows switching to one or more outputs.
/// An analogy would be the switchboard of a telephone operator.
/// Calls (inputs) come in and by re-connecting the input to an output, the operator
///   (multiplexer) can direct that input to on or more outputs.
/// One possible output is to not connect the input to anything which essentially disables
///   that input's connection to anything.
/// Not every MUX has multiple inputs; some may simply be a single input that can be enabled
///   (connected to a single output) or disabled (not connected to anything).
/// 
class aLIBEXPORT MuxClass : public EntityClass {

public:

    /// Constructor.
    MuxClass(void);

    /// Destructor.
    virtual ~MuxClass(void);

    /// Initialize the Mux Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the Mux entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Get the mux enable/disable status
    /// 
    /// \param enabled true: mux is enabled, false: the mux is disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getEnable(uint8_t* enabled);

    /// Enable the mux.
    /// 
    /// \param enable true: enables the mux for the selected channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setEnable(const uint8_t enable);

    /// Get the current selected mux channel.
    /// 
    /// \param channel Indicates which chanel is selected.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getChannel(uint8_t* channel);

    /// Set the current mux channel.
    /// 
    /// \param channel mux channel to select.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setChannel(const uint8_t channel);

    /// Get the voltage of the indicated mux channel.
    /// 
    /// \param channel The channel in which voltage was requested.
    /// \param microvolts 32 bit signed integer (in microvolts) based on the board's ground and
    ///        reference voltages.
    /// 
    /// \note Not all modules provide 32 bits of accuracy; Refer to the module's datasheet to
    ///       determine the analog bit depth and reference voltage.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getChannelVoltage(const uint8_t channel, int32_t* microvolts);

    /// Get the configuration of the mux.
    /// 
    /// \param config integer representing the mux configuration either default, or split-mode.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getConfiguration(int32_t* config);

    /// Set the configuration of the mux.
    /// 
    /// \param config integer representing the mux configuration either muxConfig_default, or
    ///        muxConfig_splitMode.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setConfiguration(const int32_t config);

    /// Get the current split mode mux configuration.
    /// 
    /// \param splitMode integer representing the channel selection for each sub-channel within
    ///        the mux.
    ///        See the data-sheet for the device for specific information.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSplitMode(int32_t* splitMode);

    /// Sets the mux's split mode configuration.
    /// 
    /// \param splitMode integer representing the channel selection for each sub-channel within
    ///        the mux.
    ///        See the data-sheet for the device for specific information.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setSplitMode(const int32_t splitMode);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_MUX_CPP_H__