/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_PointerClass_CPP.h                                //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2025 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_POINTER_CPP_H__
#define __AUTOGEN_POINTER_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// PointerClass:
/// Allows access to the reflex scratchpad from a host computer.
/// 
/// The Pointers access the pad which is a shared memory area on a BrainStem module.
/// The interface allows the use of the BrainStem scratchpad from the host, and provides a
///   mechanism for allowing the host application and BrainStem relexes to communicate.
/// 
/// The Pointer allows access to the pad in a similar manner as a file pointer accesses the
///   underlying file.
/// The cursor position can be set via setOffset. A read of a character short or int can be
///   made from that cursor position.
/// 
/// In addition the mode of the pointer can be set so that the cursor position automatically
///   increments or set so that it does not this allows for multiple reads of the same pad
///   value, or reads of multi-record values, via an incrementing pointer.
/// 
class aLIBEXPORT PointerClass : public EntityClass {

public:

    /// Constructor.
    PointerClass(void);

    /// Destructor.
    virtual ~PointerClass(void);

    /// Initialize the Pointer Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the Pointer entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Get the offset of the pointer
    /// 
    /// \param offset The value of the offset.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getOffset(uint16_t* offset);

    /// Set the offset of the pointer
    /// 
    /// \param offset The value of the offset.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setOffset(const uint16_t offset);

    /// Get the mode of the pointer
    /// 
    /// \param mode The mode: aPOINTER_MODE_STATIC or aPOINTER_MODE_AUTO_INCREMENT.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getMode(uint8_t* mode);

    /// Set the mode of the pointer
    /// 
    /// \param mode The mode: aPOINTER_MODE_STATIC or aPOINTER_MODE_AUTO_INCREMENT.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setMode(const uint8_t mode);

    /// Get the handle to the store.
    /// 
    /// \param handle The handle of the store.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getTransferStore(uint8_t* handle);

    /// Set the handle to the store.
    /// 
    /// \param handle The handle of the store.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setTransferStore(const uint8_t handle);

    /// Transfer data to the store.
    /// 
    /// \param transferLength The length of the data transfer.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr initiateTransferToStore(const uint8_t transferLength);

    /// Transfer data from the store.
    /// 
    /// \param transferLength The length of the data transfer.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr initiateTransferFromStore(const uint8_t transferLength);

    /// Get a char (1 byte) value from the pointer at this object's index, where elements are 1
    ///   byte long.
    /// 
    /// \param value The value of a single character (1 byte) stored in the pointer.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getChar(uint8_t* value);

    /// Set a char (1 byte) value to the pointer at this object's element index, where elements
    ///   are 1 byte long.
    /// 
    /// \param value The single char (1 byte) value to be stored in the pointer.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setChar(const uint8_t value);

    /// Get a short (2 byte) value from the pointer at this objects index, where elements are 2
    ///   bytes long
    /// 
    /// \param value The value of a single short (2 byte) stored in the pointer.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getShort(uint16_t* value);

    /// Set a short (2 bytes) value to the pointer at this object's element index, where elements
    ///   are 2 bytes long.
    /// 
    /// \param value The single short (2 byte) value to be set in the pointer.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setShort(const uint16_t value);

    /// Get an int (4 bytes) value from the pointer at this objects index, where elements are 4
    ///   bytes long
    /// 
    /// \param value The value of a single int (4 byte) stored in the pointer.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getInt(uint32_t* value);

    /// Set an int (4 bytes) value from the pointer at this objects index, where elements are 4
    ///   bytes long
    /// 
    /// \param value The single int (4 byte) value to be stored in the pointer.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setInt(const uint32_t value);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_POINTER_CPP_H__