/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_PortClass_CPP.h                                   //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2025 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_PORT_CPP_H__
#define __AUTOGEN_PORT_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// PortClass:
/// The Port Entity provides software control over the most basic items related to a USB Port.
/// This includes everything from the complete enable and disable of the entire port to the
///   individual control of specific pins.
/// Voltage and Current measurements are also included for devices which support the Port
///   Entity.
/// 
class aLIBEXPORT PortClass : public EntityClass {

public:

    /// Constructor.
    PortClass(void);

    /// Destructor.
    virtual ~PortClass(void);

    /// Initialize the Port Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the Port entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Gets the Vbus Voltage
    /// 
    /// \param microvolts The voltage in microvolts (1 == 1e-6V) currently present on Vbus.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVbusVoltage(int32_t* microvolts);

    /// Gets the Vbus Current
    /// 
    /// \param microamps The current in microamps (1 == 1e-6A) currently present on Vbus.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVbusCurrent(int32_t* microamps);

    /// Gets the Vconn Voltage
    /// 
    /// \param microvolts The voltage in microvolts (1 == 1e-6V) currently present on Vconn.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVconnVoltage(int32_t* microvolts);

    /// Gets the Vconn Current
    /// 
    /// \param microamps The current in microamps (1 == 1e-6A) currently present on Vconn.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVconnCurrent(int32_t* microamps);

    /// Gets the Port Power Mode: Convenience Function of get/setPortMode
    /// 
    /// \param powerMode The current power mode.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerMode(uint8_t* powerMode);

    /// Sets the Port Power Mode: Convenience Function of get/setPortMode
    /// 
    /// \param powerMode The power mode to be set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerMode(const uint8_t powerMode);

    /// Gets the current enable value of the port.
    /// 
    /// \param enable 1 = Fully enabled port; 0 = One or more disabled components.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getEnabled(uint8_t* enable);

    /// Enables or disables the entire port.
    /// 
    /// \param enable 1 = Fully enable port; 0 = Fully disable port.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setEnabled(const uint8_t enable);

    /// Gets the current enable value of the data lines.
    /// Sub-component (Data) of getEnabled.
    /// 
    /// \param enable 1 = Data enabled; 0 = Data disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataEnabled(uint8_t* enable);

    /// Enables or disables the data lines.
    /// Sub-component (Data) of setEnabled.
    /// 
    /// \param enable 1 = Enable data; 0 = Disable data.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataEnabled(const uint8_t enable);

    /// Gets the current enable value of the High Speed (HS) data lines.
    /// Sub-component of getDataEnabled.
    /// 
    /// \param enable 1 = Data enabled; 0 = Data disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataHSEnabled(uint8_t* enable);

    /// Enables or disables the High Speed (HS) data lines.
    /// Sub-component of setDataEnabled.
    /// 
    /// \param enable 1 = Enable data; 0 = Disable data.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataHSEnabled(const uint8_t enable);

    /// Gets the current enable value of the High Speed A side (HSA) data lines.
    /// Sub-component of getDataHSEnabled.
    /// 
    /// \param enable 1 = Data enabled; 0 = Data disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataHS1Enabled(uint8_t* enable);

    /// Enables or disables the High Speed A side (HSA) data lines.
    /// Sub-component of setDataHSEnabled.
    /// 
    /// \param enable 1 = Enable data; 0 = Disable data.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataHS1Enabled(const uint8_t enable);

    /// Gets the current enable value of the High Speed B side (HSB) data lines.
    /// Sub-component of getDataHSEnabled.
    /// 
    /// \param enable 1 = Data enabled; 0 = Data disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataHS2Enabled(uint8_t* enable);

    /// Enables or disables the High Speed B side (HSB) data lines.
    /// Sub-component of setDataHSEnabled.
    /// 
    /// \param enable 1 = Enable data; 0 = Disable data.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataHS2Enabled(const uint8_t enable);

    /// Gets the current enable value of the Super Speed (SS) data lines.
    /// Sub-component of getDataEnabled.
    /// 
    /// \param enable 1 = Data enabled; 0 = Data disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataSSEnabled(uint8_t* enable);

    /// Enables or disables the Super Speed (SS) data lines.
    /// Sub-component of setDataEnabled.
    /// 
    /// \param enable 1 = Enable data; 0 = Disable data.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataSSEnabled(const uint8_t enable);

    /// Gets the current enable value of the Super Speed A side (SSA) data lines.
    /// Sub-component of getDataSSEnabled.
    /// 
    /// \param enable 1 = Data enabled; 0 = Data disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataSS1Enabled(uint8_t* enable);

    /// Enables or disables the Super Speed A side (SSA) data lines.
    /// Sub-component of setDataEnabled.
    /// 
    /// \param enable 1 = Enable data; 0 = Disable data.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataSS1Enabled(const uint8_t enable);

    /// Gets the current enable value of the Super Speed B side (SSB) data lines.
    /// Sub-component of getDataSSEnabled.
    /// 
    /// \param enable 1 = Data enabled; 0 = Data disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataSS2Enabled(uint8_t* enable);

    /// Enables or disables the Super Speed B side (SSB) data lines.
    /// Sub-component of setDataSSEnabled.
    /// 
    /// \param enable 1 = Enable data; 0 = Disable data.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataSS2Enabled(const uint8_t enable);

    /// Gets the current enable value of the power lines. Sub-component (Power) of getEnabled.
    /// 
    /// \param enable 1 = Power enabled; 0 = Power disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerEnabled(uint8_t* enable);

    /// Enables or Disables the power lines. Sub-component (Power) of setEnable.
    /// 
    /// \param enable 1 = Enable power; 0 = Disable disable.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerEnabled(const uint8_t enable);

    /// Gets the Port Data Role.
    /// 
    /// \param dataRole The data role to be set. See datasheet for details.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataRole(uint8_t* dataRole);

    /// Gets the current enable value of the Vconn lines.
    /// Sub-component (Vconn) of getEnabled.
    /// 
    /// \param enable 1 = Vconn enabled; 0 = Vconn disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVconnEnabled(uint8_t* enable);

    /// Enables or disables the Vconn lines.
    /// Sub-component (Vconn) of setEnabled.
    /// 
    /// \param enable 1 = Enable Vconn lines; 0 = Disable Vconn lines.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setVconnEnabled(const uint8_t enable);

    /// Gets the current enable value of the Vconn1 lines.
    /// Sub-component of getVconnEnabled.
    /// 
    /// \param enable 1 = Vconn1 enabled; 0 = Vconn1 disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVconn1Enabled(uint8_t* enable);

    /// Enables or disables the Vconn1 lines.
    /// Sub-component of setVconnEnabled.
    /// 
    /// \param enable 1 = Enable Vconn1 lines; 0 = Disable Vconn1 lines.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setVconn1Enabled(const uint8_t enable);

    /// Gets the current enable value of the Vconn2 lines.
    /// Sub-component of getVconnEnabled.
    /// 
    /// \param enable 1 = Vconn2 enabled; 0 = Vconn2 disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVconn2Enabled(uint8_t* enable);

    /// Enables or disables the Vconn2 lines.
    /// Sub-component of setVconnEnabled.
    /// 
    /// \param enable 1 = Enable Vconn2 lines; 0 = Disable Vconn2 lines.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setVconn2Enabled(const uint8_t enable);

    /// Gets the current enable value of the CC lines.
    /// Sub-component (CC) of getEnabled.
    /// 
    /// \param enable 1 = CC enabled; 0 = CC disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCCEnabled(uint8_t* enable);

    /// Enables or disables the CC lines.
    /// Sub-component (CC) of setEnabled.
    /// 
    /// \param enable 1 = Enable CC lines; 0 = Disable CC lines.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCCEnabled(const uint8_t enable);

    /// Gets the current enable value of the CC1 lines.
    /// Sub-component of getCCEnabled.
    /// 
    /// \param enable 1 = CC1 enabled; 0 = CC1 disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC1Enabled(uint8_t* enable);

    /// Enables or disables the CC1 lines.
    /// Sub-component of setCCEnabled.
    /// 
    /// \param enable 1 = Enable CC1 lines; 0 = Disable CC1 lines.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCC1Enabled(const uint8_t enable);

    /// Gets the current enable value of the CC2 lines.
    /// Sub-component of getCCEnabled.
    /// 
    /// \param enable 1 = CC2 enabled; 0 = CC2 disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC2Enabled(uint8_t* enable);

    /// Enables or disables the CC2 lines.
    /// Sub-component of setCCEnabled.
    /// 
    /// \param enable 1 = Enable CC2 lines; 0 = Disable CC2 lines.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCC2Enabled(const uint8_t enable);

    /// Gets the current voltage setpoint value for the port.
    /// 
    /// \param value the voltage setpoint of the port in uV.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVoltageSetpoint(uint32_t* value);

    /// Sets the current voltage setpoint value for the port.
    /// 
    /// \param value the voltage setpoint of the port in uV.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setVoltageSetpoint(const uint32_t value);

    /// A bit mapped representation of the current state of the port.
    /// Reflects what he port IS which may differ from what was requested.
    /// 
    /// \param state Variable to be filled with the current state.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getState(uint32_t* state);

    /// Gets the speed of the enumerated device.
    /// 
    /// \param speed Bit mapped value representing the devices speed.
    ///        See "Devices" reference for details.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataSpeed(uint8_t* speed);

    /// Gets current mode of the port
    /// 
    /// \param mode Bit mapped value representing the ports mode.
    ///        See "Devices" reference for details.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getMode(uint32_t* mode);

    /// Sets the mode of the port
    /// 
    /// \param mode Port mode to be set. See "Devices" documentation for details.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setMode(const uint32_t mode);

    /// Returns any errors that are present on the port.
    /// Calling this function will clear the current errors. If the error persists it will be set
    ///   again.
    /// 
    /// \param errors Bit mapped field representing the current errors of the ports
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getErrors(uint32_t* errors);

    /// Gets the current limit of the port.
    /// 
    /// \param limit Variable to be filled with the limit in microAmps (uA).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCurrentLimit(uint32_t* limit);

    /// Sets the current limit of the port.
    /// 
    /// \param limit Current limit to be applied in microAmps (uA).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCurrentLimit(const uint32_t limit);

    /// Gets the current limit mode.
    /// The mode determines how the port will react to an over current condition.
    /// 
    /// \param mode Variable to be filled with an enumerated representation of the current limit
    ///        mode.
    ///        Available modes are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCurrentLimitMode(uint8_t* mode);

    /// Sets the current limit mode.
    /// The mode determines how the port will react to an over current condition.
    /// 
    /// \param mode An enumerated representation of the current limit mode.
    ///        Available modes are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCurrentLimitMode(const uint8_t mode);

    /// Gets the current available power.
    /// This value is determined by the power manager which is responsible for budgeting the
    ///   systems available power envelope.
    /// 
    /// \param power Variable to be filled with the available power in milli-watts (mW).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getAvailablePower(uint32_t* power);

    /// Gets the currently allocated power
    /// This value is determined by the power manager which is responsible for budgeting the
    ///   systems available power envelope.
    /// 
    /// \param power Variable to be filled with the allocated power in milli-watts (mW).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getAllocatedPower(int32_t* power);

    /// Gets the user defined power limit for the port.
    /// 
    /// \param limit Variable to be filled with the power limit in milli-watts (mW).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerLimit(uint32_t* limit);

    /// Sets a user defined power limit for the port.
    /// 
    /// \param limit Power limit to be applied in milli-watts (mW).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerLimit(const uint32_t limit);

    /// Gets the power limit mode.
    /// The mode determines how the port will react to an over power condition.
    /// 
    /// \param mode Variable to be filled with an enumerated representation of the power limit
    ///        mode.
    ///        Available modes are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerLimitMode(uint8_t* mode);

    /// Sets the power limit mode.
    /// The mode determines how the port will react to an over power condition.
    /// 
    /// \param mode An enumerated representation of the power limit mode to be applied
    ///        Available modes are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerLimitMode(const uint8_t mode);

    /// Gets a user defined name of the port.
    /// Helpful for identifying ports/devices in a static environment.
    /// 
    /// \param buffer pointer to the start of a c style buffer to be filled
    /// \param bufferLength Length of the buffer to be filed
    /// \param unloadedLength Length that was actually received and filled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getName(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Sets a user defined name of the port.
    /// Helpful for identifying ports/devices in a static environment.
    /// 
    /// \param buffer Pointer to the start of a c style buffer to be transferred.
    /// \param bufferLength Length of the buffer to be transferred.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setName(const uint8_t* buffer, const size_t bufferLength);

    /// Gets the CC Current Limit Resistance
    /// The CC Current limit is the value that's set for the pull up resistance on the CC lines
    ///   for basic USB-C negotations.
    /// 
    /// \param value Variable to be filled with an enumerated representation of the CC Current
    ///        limit.
    ///            - 0 = None
    ///            - 1 = Default (500/900mA)
    ///            - 2 = 1.5A
    ///            - 3 = 3.0A
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCCCurrentLimit(uint8_t* value);

    /// Sets the CC Current Limit Resistance
    /// The CC Current limit is the value that's set for the pull up resistance on the CC lines
    ///   for basic USB-C negotations.
    /// 
    /// \param value Variable to be filled with an enumerated representation of the CC Current
    ///        limit.
    ///            - 0 = None
    ///            - 1 = Default (500/900mA)
    ///            - 2 = 1.5A
    ///            - 3 = 3.0A
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCCCurrentLimit(const uint8_t value);

    /// Gets the HighSpeed Data Routing Behavior.
    /// The mode determines how the port will route the data lines.
    /// 
    /// \param mode Variable to be filled with an enumerated representation of the routing
    ///        behavior.
    ///        Available modes are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataHSRoutingBehavior(uint8_t* mode);

    /// Sets the HighSpeed Data Routing Behavior.
    /// The mode determines how the port will route the data lines.
    /// 
    /// \param mode An enumerated representation of the routing behavior.
    ///        Available modes are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataHSRoutingBehavior(const uint8_t mode);

    /// Gets the SuperSpeed Data Routing Behavior.
    /// The mode determines how the port will route the data lines.
    /// 
    /// \param mode Variable to be filled with an enumerated representation of the routing
    ///        behavior.
    ///        Available modes are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataSSRoutingBehavior(uint8_t* mode);

    /// Sets the SuperSpeed Data Routing Behavior.
    /// The mode determines how the port will route the data lines.
    /// 
    /// \param mode An enumerated representation of the routing behavior.
    ///        Available modes are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataSSRoutingBehavior(const uint8_t mode);

    /// Gets the Vbus Accumulated Power
    /// 
    /// \param milliwatthours The accumulated power on Vbus in milliwatt-hours.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVbusAccumulatedPower(int32_t* milliwatthours);

    /// Sets the Vbus Accumulated Power
    /// 
    /// \param milliwatthours The accumulated power on Vbus in milliwatt-hours to be set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setVbusAccumulatedPower(const int32_t milliwatthours);

    /// Resets the Vbus Accumulated Power to zero.
    /// 
    /// \deprecated Replace with setVbusAccumulatedPower(0)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    DEPRECATED("Replace with setVbusAccumulatedPower(0)")
    aErr resetVbusAccumulatedPower(void);

    /// Gets the Vconn Accumulated Power
    /// 
    /// \param milliwatthours The accumulated power on Vconn in milliwatt-hours.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVconnAccumulatedPower(int32_t* milliwatthours);

    /// Sets the Vconn Accumulated Power
    /// 
    /// \param milliwatthours The accumulated power on Vconn to be set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setVconnAccumulatedPower(const int32_t milliwatthours);

    /// Resets the Vconn Accumulated Power to zero.
    /// 
    /// \deprecated Replace with setVconnAccumulatedPower(0)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    DEPRECATED("Replace with setVconnAccumulatedPower(0)")
    aErr resetVconnAccumulatedPower(void);

    /// Sets the ports USB 2.0 High Speed Boost Settings
    /// The setting determines how much additional drive the USB 2.0 signal will have in High
    ///   Speed mode.
    /// 
    /// \param boost An enumerated representation of the boost range.
    ///        Available value are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setHSBoost(const uint8_t boost);

    /// Gets the ports USB 2.0 High Speed Boost Settings
    /// The setting determines how much additional drive the USB 2.0 signal will have in High
    ///   Speed mode.
    /// 
    /// \param boost An enumerated representation of the boost range.
    ///        Available modes are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getHSBoost(uint8_t* boost);

    /// Gets the current CC1 Strapping on local and remote
    /// The state is a bit packed value where the upper byte is used to represent the remote or
    ///   partner device attached to the ports resistance and the lower byte is used to represent
    ///   the local or hubs resistance.
    /// 
    /// \param value Variable to be filled with an packed enumerated representation of the CC
    ///        state.
    ///        Enumeration values for each byte are as follows:
    ///            - None = 0 = portCC1State_None
    ///            - Invalid = 1 = portCC1State_Invalid
    ///            - Rp (default) = 2 = portCC1State_RpDefault
    ///            - Rp (1.5A) = 3 = portCC1State_Rp1p5
    ///            - Rp (3A) = 4 = portCC1State_Rp3p0
    ///            - Rd = 5 = portCC1State_Rd
    ///            - Ra = 6 = portCC1State_Ra
    ///            - Managed by controller = 7 = portCC1State_Managed
    ///            - Unknown = 8 = portCC1State_Unknown
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC1State(uint16_t* value);

    /// Gets the current CC2 Strapping on local and remote
    /// The state is a bit packed value where the upper byte is used to represent the remote or
    ///   partner device attached to the ports resistance and the lower byte is used to represent
    ///   the local or hubs resistance.
    /// 
    /// \param value Variable to be filled with an packed enumerated representation of the CC
    ///        state.
    ///        Enumeration values for each byte are as follows:
    ///            - None = 0 = portCC2State_None
    ///            - Invalid = 1 = portCC2State_Invalid
    ///            - Rp (default) = 2 = portCC2State_RpDefault
    ///            - Rp (1.5A) = 3 = portCC2State_Rp1p5
    ///            - Rp (3A) = 4 = portCC2State_Rp3p0
    ///            - Rd = 5 = portCC2State_Rd
    ///            - Ra = 6 = portCC2State_Ra
    ///            - Managed by controller = 7 = portCC2State_Managed
    ///            - Unknown = 8 = portCC2State_Unknown
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC2State(uint16_t* value);

    /// Get the voltage of SBU1 for a port.
    /// 
    /// \param microvolts The USB channel voltage in micro-volts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSBU1Voltage(int32_t* microvolts);

    /// Get the voltage of SBU2 for a port.
    /// 
    /// \param microvolts The USB channel voltage in micro-volts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSBU2Voltage(int32_t* microvolts);

    /// Get the voltage of CC1 for a port.
    /// 
    /// \param microvolts The USB channel voltage in micro-volts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC1Voltage(int32_t* microvolts);

    /// Get the voltage of CC2 for a port.
    /// 
    /// \param microvolts The USB channel voltage in micro-volts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC2Voltage(int32_t* microvolts);

    /// Get the current through the CC1 for a port.
    /// 
    /// \param microamps The USB channel current in micro-amps (1 == 1e-6A).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC1Current(int32_t* microamps);

    /// Get the current through the CC2 for a port.
    /// 
    /// \param microamps The USB channel current in micro-amps (1 == 1e-6A).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC2Current(int32_t* microamps);

    /// Gets the CC1 Accumulated Power
    /// 
    /// \param milliwatthours The accumulated power on Vconn in milliwatt-hours.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC1AccumulatedPower(int32_t* milliwatthours);

    /// Sets the CC1 Accumulated Power
    /// 
    /// \param milliwatthours The accumulated power on Vconn to be set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCC1AccumulatedPower(const int32_t milliwatthours);

    /// Gets the CC2 Accumulated Power
    /// 
    /// \param milliwatthours The accumulated power on Vconn in milliwatt-hours.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC2AccumulatedPower(int32_t* milliwatthours);

    /// Sets the CC2 Accumulated Power
    /// 
    /// \param milliwatthours The accumulated power on Vconn to be set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCC2AccumulatedPower(const int32_t milliwatthours);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_PORT_CPP_H__