/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_RailClass_CPP.h                                   //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2025 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_RAIL_CPP_H__
#define __AUTOGEN_RAIL_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// RailClass:
/// Provides power rail functionality on certain modules.
/// The RailClass can be used to control power to downstream devices.
/// It has the ability to take current and voltage measurements, and depending on hardware,
///   may have additional modes and capabilities.
/// 
class aLIBEXPORT RailClass : public EntityClass {

public:

    /// Constructor.
    RailClass(void);

    /// Destructor.
    virtual ~RailClass(void);

    /// Initialize the Rail Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the Rail entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Get the rail current.
    /// 
    /// \param microamps The current in micro-amps (1 == 1e-6A).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCurrent(int32_t* microamps);

    /// Set the rail supply current. Rail current control capabilities vary between modules.
    /// Refer to the module datasheet for definition of the rail current capabilities.
    /// 
    /// \param microamps The current in micro-amps (1 == 1e-6A) to be supply by the rail.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCurrentSetpoint(const int32_t microamps);

    /// Get the rail setpoint current. Rail current control capabilities vary between modules.
    /// Refer to the module datasheet for definition of the rail current capabilities.
    /// 
    /// \param microamps The current in micro-amps (1 == 1e-6A) the rail is trying to achieve.
    ///        On some modules this is a measured value so it may not exactly match what was
    ///        previously set via the setCurrent interface.
    ///        Refer to the module datasheet to to determine if this is a measured or stored
    ///        value.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCurrentSetpoint(int32_t* microamps);

    /// Set the rail current limit setting. (Check product datasheet to see if this feature is
    ///   available)
    /// 
    /// \param microamps The current in micro-amps (1 == 1e-6A).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCurrentLimit(const int32_t microamps);

    /// Get the rail current limit setting. (Check product datasheet to see if this feature is
    ///   available)
    /// 
    /// \param microamps The current in micro-amps (1 == 1e-6A).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCurrentLimit(int32_t* microamps);

    /// Get the rail temperature.
    /// 
    /// \param microcelsius The measured temperature associated with the rail in micro-Celsius (1
    ///        == 1e-6˚C).
    ///        The temperature may be associated with the module's internal rail circuitry or an
    ///        externally connected temperature sensors.
    ///        Refer to the module datasheet for definition of the temperature measurement
    ///        location and specific capabilities.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getTemperature(int32_t* microcelsius);

    /// Get the state of the external rail switch. Not all rails can be switched on and off.
    /// Refer to the module datasheet for capability specification of the rails.
    /// 
    /// \param enable true: enabled: connected to the supply rail voltage;
    ///        false: disabled: disconnected from the supply rail voltage
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getEnable(uint8_t* enable);

    /// Set the state of the external rail switch.
    /// Not all rails can be switched on and off. Refer to the module datasheet for capability
    ///   specification of the rails.
    /// 
    /// \param enable true: enable and connect to the supply rail voltage;
    ///        false: disable and disconnect from the supply rail voltage
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setEnable(const uint8_t enable);

    /// Get the rail supply voltage.
    /// Rail voltage control capabilities vary between modules. Refer to the module datasheet for
    ///   definition of the rail voltage capabilities.
    /// 
    /// \param microvolts The voltage in micro-volts (1 == 1e-6V) currently supplied by the rail.
    ///        On some modules this is a measured value so it may not exactly match what was
    ///        previously set via the setVoltage interface.
    ///        Refer to the module datasheet to determine if this is a measured or stored value.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVoltage(int32_t* microvolts);

    /// Set the rail supply voltage.
    /// Rail voltage control capabilities vary between modules. Refer to the module datasheet for
    ///   definition of the rail voltage capabilities.
    /// 
    /// \param microvolts The voltage in micro-volts (1 == 1e-6V) to be supplied by the rail.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setVoltageSetpoint(const int32_t microvolts);

    /// Get the rail setpoint voltage.
    /// Rail voltage control capabilities vary between modules. Refer to the module datasheet for
    ///   definition of the rail voltage capabilities.
    /// 
    /// \param microvolts The voltage in micro-volts (1 == 1e-6V) the rail is trying to achieve.
    ///        On some modules this is a measured value so it may not exactly match what was
    ///        previously set via the setVoltage interface.
    ///        Refer to the module datasheet to determine if this is a measured or stored value.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVoltageSetpoint(int32_t* microvolts);

    /// Set the rail voltage minimum limit setting. (Check product datasheet to see if this
    ///   feature is available)
    /// 
    /// \param microvolts The voltage in micro-volts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setVoltageMinLimit(const int32_t microvolts);

    /// Get the rail voltage minimum limit setting. (Check product datasheet to see if this
    ///   feature is available)
    /// 
    /// \param microvolts The voltage in micro-volts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVoltageMinLimit(int32_t* microvolts);

    /// Set the rail voltage maximum limit setting. (Check product datasheet to see if this
    ///   feature is available)
    /// 
    /// \param microvolts The voltage in micro-volts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setVoltageMaxLimit(const int32_t microvolts);

    /// Get the rail voltage maximum limit setting. (Check product datasheet to see if this
    ///   feature is available)
    /// 
    /// \param microvolts The voltage in micro-volts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getVoltageMaxLimit(int32_t* microvolts);

    /// Get the rail supply power.
    /// Rail power control capabilities vary between modules. Refer to the module datasheet for
    ///   definition of the rail power capabilities.
    /// 
    /// \param milliwatts The power in milli-watts (1 == 1e-3W) currently supplied by the rail.
    ///        On some modules this is a measured value so it may not exactly match what was
    ///        previously set via the setPower interface.
    ///        Refer to the module datasheet to determine if this is a measured or stored value.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPower(int32_t* milliwatts);

    /// Set the rail supply power.
    /// Rail power control capabilities vary between modules. Refer to the module datasheet for
    ///   definition of the rail power capabilities.
    /// 
    /// \param milliwatts The power in milli-watts (1 == 1e-3W) to be supplied by the rail.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerSetpoint(const int32_t milliwatts);

    /// Get the rail setpoint power.
    /// Rail power control capabilities vary between modules. Refer to the module datasheet for
    ///   definition of the rail power capabilities.
    /// 
    /// \param milliwatts The power in milli-watts (1 == 1e-3W) the rail is trying to achieve.
    ///        On some modules this is a measured value so it may not exactly match what was
    ///        previously set via the setPower interface.
    ///        Refer to the module datasheet to determine if this is a measured or stored value.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerSetpoint(int32_t* milliwatts);

    /// Set the rail power maximum limit setting. (Check product datasheet to see if this feature
    ///   is available)
    /// 
    /// \param milliwatts The power in milli-watts (mW).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerLimit(const int32_t milliwatts);

    /// Get the rail power maximum limit setting. (Check product datasheet to see if this feature
    ///   is available)
    /// 
    /// \param milliwatts The power in milli-watts (mW).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerLimit(int32_t* milliwatts);

    /// Get the rail load resistance.
    /// Rail resistance control capabilities vary between modules. Refer to the module datasheet
    ///   for definition of the rail resistance capabilities.
    /// 
    /// \param milliohms The resistance in milli-ohms (1 == 1e-3Ohms) currently drawn by the rail.
    ///        On some modules this is a measured value so it may not exactly match what was
    ///        previously set via the setResistance interface.
    ///        Refer to the module datasheet to determine if this is a measured or stored value.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getResistance(int32_t* milliohms);

    /// Set the rail load resistance.
    /// Rail resistance control capabilities vary between modules. Refer to the module datasheet
    ///   for definition of the rail resistance capabilities.
    /// 
    /// \param milliohms The resistance in milli-ohms (1 == 1e-3Ohms) to be drawn by the rail.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setResistanceSetpoint(const int32_t milliohms);

    /// Get the rail setpoint resistance.
    /// Rail resistance control capabilities vary between modules. Refer to the module datasheet
    ///   for definition of the rail resistance capabilities.
    /// 
    /// \param milliohms The resistance in milli-ohms (1 == 1e-3Ohms) the rail is trying to
    ///        achieve.
    ///        On some modules this is a measured value so it may not exactly match what was
    ///        previously set via the setResistance interface.
    ///        Refer to the module datasheet to to determine if this is a measured or stored
    ///        value.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getResistanceSetpoint(int32_t* milliohms);

    /// Enable or Disable kelvin sensing on the module.
    /// Refer to the module datasheet for definition of the rail kelvin sensing capabilities.
    /// 
    /// \param enable enable or disable kelvin sensing.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setKelvinSensingEnable(const uint8_t enable);

    /// Determine whether kelvin sensing is enabled or disabled.
    /// Refer to the module datasheet for definition of the rail kelvin sensing capabilities.
    /// 
    /// \param enable Kelvin sensing is enabled or disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getKelvinSensingEnable(uint8_t* enable);

    /// Determine whether kelvin sensing has been disabled by the system.
    /// Refer to the module datasheet for definition of the rail kelvin sensing capabilities.
    /// 
    /// \param state Kelvin sensing is enabled or disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getKelvinSensingState(uint8_t* state);

    /// Set the operational mode of the rail.
    /// Refer to the module datasheet for definition of the rail operational capabilities.
    /// 
    /// \param mode The operational mode to employ.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setOperationalMode(const uint8_t mode);

    /// Determine the current operational mode of the system.
    /// Refer to the module datasheet for definition of the rail operational mode capabilities.
    /// 
    /// \param mode The current operational mode setting.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getOperationalMode(uint8_t* mode);

    /// Determine the current operational state of the system.
    /// Refer to the module datasheet for definition of the rail operational states.
    /// 
    /// \param state The current operational state, hardware configuration, faults, and operating
    ///        mode.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getOperationalState(uint32_t* state);

    /// Clears the current fault state of the rail.
    /// Refer to the module datasheet for definition of the rail faults.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr clearFaults(void);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_RAIL_CPP_H__