//
//  bs_pd_packet.h
//
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright 1994-2024. Acroname Inc.                              //
//                                                                 //
// This software is the property of Acroname Inc.  Any             //
// distribution, sale, transmission, or re-use of this code is     //
// strictly forbidden except with permission from Acroname Inc.    //
//                                                                 //
// To the full extent allowed by law, Acroname Inc. also excludes  //
// for itself and its suppliers any liability, whether based in    //
// contract or tort (including negligence), for direct,            //
// incidental, consequential, indirect, special, or punitive       //
// damages of any kind, or for loss of revenue or profits, loss of //
// business, loss of information or data, or other financial loss  //
// arising out of or in connection with this software, even if     //
// Acroname Inc. has been advised of the possibility of such       //
// damages.                                                        //
//                                                                 //
// Acroname Inc.                                                   //
// www.acroname.com                                                //
// 720-564-0373                                                    //
//                                                                 //
/////////////////////////////////////////////////////////////////////
#ifndef BS_PD_PACKET_H
#define BS_PD_PACKET_H

#include <stdint.h>

#if defined(__cplusplus)
#include <vector>
#endif//defined(__cplusplus)

/////////////////////////////////////////////////////////////////////
/// BrainStem Power Delivery Packet

/** \defgroup BSPDPacket BrainStem Power Delivery Packet.
 * \ref BS_PD_Packet "bs_pd_packet.h" provides the packet structure which
 * is used with the PDChannelLogger interface.
 */


#ifdef _WIN32
#ifndef aUNUSED
#define aUNUSED
#endif
#else
#ifndef aUNUSED
#define aUNUSED             __attribute((unused))
#endif
#endif

/** Packet Direction enumeration */
typedef enum BS_PD_Packet_Direction {
    kBS_PD_Packet_Direction_Invalid,
    kBS_PD_Packet_Direction_Transmit,
    kBS_PD_Packet_Direction_Receive,
    kBS_PD_Packet_Direction_Sniff,

    kBS_PD_Packet_Direction_UNKNOWN,
    kBS_PD_Packet_Direction_LAST
} BS_PD_Packet_Direction_t;

/** Packet Direction strings */
static const char* BS_PDLog_Packet_Direction_Strings[kBS_PD_Packet_Direction_LAST] aUNUSED = {
    "Invalid",
    "TX",
    "RX",
    "Sniff", //through, sniff, RTX
    "Unknown"
};


/** Packet CC Channel enumeration */
typedef enum BS_PD_Packet_CC_Channel {
    kBS_PD_Packet_CC_Channel_Invalid,
    kBS_PD_Packet_CC_Channel_CC1,
    kBS_PD_Packet_CC_Channel_CC2,

    kBS_PD_Packet_CC_Channel_UNKNOWN,
    kBS_PD_Packet_CC_Channel_LAST
} BS_PD_Packet_CC_Channel_t;

/** Packet CC Channel strings */
static const char* BS_PDLog_Packet_CC_Channel_Strings[kBS_PD_Packet_CC_Channel_LAST] aUNUSED = {
    "Invalid",
    "CC1",
    "CC2",
    "Unknown"
};

/** Packet SOP type enumeration */
typedef enum BS_PD_Packet_SOP {
    kBS_PD_Packet_SOP,
    kBS_PD_Packet_SOP_P,
    kBS_PD_Packet_SOP_P_P,
    kBS_PD_Packet_SOP_P_DEBUG,
    kBS_PD_Packet_SOP_P_P_DEBUG,

    kBS_PD_Packet_SOP_UNKNOWN,
    kBS_PD_Packet_SOP_LAST
} BS_PD_Packet_SOP_t;

/** Packet SOP type strings */
static const char* BS_PD_Packet_SOP_Strings[kBS_PD_Packet_SOP_LAST] aUNUSED = {
    "SOP",
    "SOP\'",
    "SOP\'\'",
    "SOP\' Debug",
    "SOP\'\' Debug",
    "Unknown"
};


#if defined(__cplusplus)

/** BS_PD_Packet structure - Contains information describing the contained Power Delivery Packet */
typedef struct BS_PD_Packet {
    
    BS_PD_Packet() :
        channel(0),
        seconds(0),
        uSeconds(0),
        direction(BS_PD_Packet_Direction_t::kBS_PD_Packet_Direction_UNKNOWN),
        sop(BS_PD_Packet_SOP_t::kBS_PD_Packet_SOP_UNKNOWN),
        payload(std::vector<uint8_t>()),
        event(0),
        ccChannel(BS_PD_Packet_CC_Channel_t::kBS_PD_Packet_CC_Channel_UNKNOWN),
        crc(0)
        {}

    BS_PD_Packet(uint8_t channel,
                 uint32_t seconds,
                 uint32_t uSeconds,
                 BS_PD_Packet_Direction_t direction,
                 BS_PD_Packet_SOP_t sop,
                 std::vector<uint8_t> payload,
                 uint32_t event = 0, // 0 = pdEventNone
                 BS_PD_Packet_CC_Channel_t ccChannel = BS_PD_Packet_CC_Channel_t::kBS_PD_Packet_CC_Channel_UNKNOWN,
                 uint32_t crc = 0)
                 :
                 channel(channel),
                 seconds(seconds),
                 uSeconds(uSeconds),
                 direction(direction),
                 sop(sop),
                 payload(payload),
                 event(event),
                 ccChannel(ccChannel),
                 crc(crc)        {}

    uint8_t channel; /**< PD Channel this packet refers too.*/
    uint32_t seconds; /**< Seconds since device boot*/
    uint32_t uSeconds; /**< Micro seconds since device boot*/
    BS_PD_Packet_Direction_t direction; /**< PD Communication direction*/
    BS_PD_Packet_SOP_t sop; /**< PD SOP type*/
    std::vector<uint8_t> payload; /**< Raw PD packet*/
    uint32_t event; /**< PD Event type - defined in aProtocolDefs.h*/
    BS_PD_Packet_CC_Channel_t ccChannel; /**< CC Channel type */
    uint32_t crc; /**< CRC of payload */
    
} BS_PD_Packet_t;

#endif//defined(__cplusplus)


#endif // BS_PD_PACKET_H
