/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_EthernetClass_CPP.h                               //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2026 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_ETHERNET_CPP_H__
#define __AUTOGEN_ETHERNET_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// EthernetClass:
/// IP configuration.  MAC info.  BrainD port.
/// 
class aLIBEXPORT EthernetClass : public EntityClass {

public:

    /// Constructor.
    EthernetClass(void);

    /// Destructor.
    virtual ~EthernetClass(void);

    /// Initialize the Ethernet Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the Ethernet entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Sets the Ethernet's interface to enabled/disabled.
    /// 
    /// \param enabled 1 = enabled; 0 = disabled
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setEnabled(const uint8_t enabled);

    /// Gets the current enable value of the Ethernet interface.
    /// 
    /// \param enabled 1 = Fully enabled network connectivity; 0 = Ethernet MAC is disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getEnabled(uint8_t* enabled);

    /// Get the method in which IP Address is assigned to this device
    /// 
    /// \param addressStyle Method used.  Current methods
    ///          - NONE = 0
    ///          - STATIC = 1
    ///          - DHCP = 2
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getNetworkConfiguration(uint8_t* addressStyle);

    /// Get the method in which IP Address is assigned to this device
    /// 
    /// \param addressStyle Method to use.  See getNetworkConfiguration for addressStyle
    ///        enumerations.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setNetworkConfiguration(const uint8_t addressStyle);

    /// Get the expected IPv4 address of this device, when networkConfiguration == STATIC
    /// 
    /// \param buffer alias to an array of uint8_t[4] for returned output
    /// \param bufferLength size of buffer.  Should be 4.
    /// \param unloadedLength occupied bytes in buffer, Should be 4 post-call.
    /// 
    /// \note The functional IPv4 address of The Module will differ if NetworkConfiguration !=
    ///       STATIC.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getStaticIPv4Address(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Set the desired IPv4 address of this device, if NetworkConfiguration == STATIC.
    /// 
    /// \param buffer alias to an array of uint8_t[4] with an IP address
    /// \param bufferLength size of buffer.  Should be 4.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setStaticIPv4Address(const uint8_t* buffer, const size_t bufferLength);

    /// Get the expected IPv4 netmask of this device, when networkConfiguration == STATIC
    /// 
    /// \param buffer alias to an array of uint8_t[4] for returned output
    /// \param bufferLength size of buffer.  Should be 4.
    /// \param unloadedLength occupied bytes in buffer, Should be 4 post-call.
    /// 
    /// \note The functional IPv4 netmask of The Module will differ if NetworkConfiguration !=
    ///       STATIC.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getStaticIPv4Netmask(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Set the desired IPv4 address of this device, if NetworkConfiguration == STATIC
    /// 
    /// \param buffer alias to an array of uint8_t[4] with an IP address
    /// \param bufferLength size of buffer.  Should be 4.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setStaticIPv4Netmask(const uint8_t* buffer, const size_t bufferLength);

    /// Get the expected IPv4 gateway of this device, when networkConfiguration == STATIC
    /// 
    /// \param buffer alias to an array of uint8_t[4] for returned output
    /// \param bufferLength size of buffer.  Should be 4.
    /// \param unloadedLength occupied bytes in buffer, Should be 4 post-call.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getStaticIPv4Gateway(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Set the desired IPv4 gateway of this device, if NetworkConfiguration == STATIC
    /// 
    /// \param buffer alias to an array of uint8_t[4] with an IP address
    /// \param bufferLength size of buffer.  Should be 4.
    ///        setStaticIPv4Gateway([192, 168, 1, 1], 4) would equate with address "192.168.1.1"
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setStaticIPv4Gateway(const uint8_t* buffer, const size_t bufferLength);

    /// Get the effective IP address of this device.
    /// 
    /// \param buffer alias to an array of uint8_t[4] for returned output
    /// \param bufferLength size of buffer.  Should be 4.
    /// \param unloadedLength occupied bytes in buffer, Should be 4 post-call.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getIPv4Address(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Get the effective IP netmask of this device.
    /// 
    /// \param buffer alias to an array of uint8_t[4] for returned output
    /// \param bufferLength size of buffer.  Should be 4.
    /// \param unloadedLength occupied bytes in buffer, Should be 4 post-call.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getIPv4Netmask(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Get the effective IP gateway of this device.
    /// 
    /// \param buffer alias to an array of uint8_t[4] for returned output
    /// \param bufferLength size of buffer.  Should be 4.
    /// \param unloadedLength occupied bytes in buffer, Should be 4 post-call.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getIPv4Gateway(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Set IPv4 DNS Addresses (plural), if NetworkConfiguration == STATIC
    /// 
    /// \param buffer alias to an array of uint8_t[N][4]
    /// \param bufferLength Total array length in bytes.  Must be a multiple of 4.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setStaticIPv4DNSAddress(const uint8_t* buffer, const size_t bufferLength);

    /// Get IPv4 DNS addresses (plural), when NetworkConfiguration == STATIC
    /// 
    /// \param buffer alias to an array of uint8_t[N][4]
    /// \param bufferLength Maximum length of array, in bytes.
    /// \param unloadedLength Length of occupied bytes of buffer, after the call.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getStaticIPv4DNSAddress(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Get effective IPv4 DNS addresses, for the current NetworkConfiguration
    /// 
    /// \param buffer alias to an array of uint8_t[N][4]
    /// \param bufferLength Maximum length of array, in bytes.
    /// \param unloadedLength Length of occupied bytes of buffer, after the call.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getIPv4DNSAddress(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Set hostname that's requested when this device sends a DHCP request.
    /// 
    /// \param buffer alias to an array of uint8_t[N]
    /// \param bufferLength N, for N bytes.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setHostname(const uint8_t* buffer, const size_t bufferLength);

    /// Get hostname that's requested when this device sends a DHCP request.
    /// 
    /// \param buffer alias to an array of uint8_t[N]
    /// \param bufferLength N, for N bytes.
    /// \param unloadedLength Length of occupied bytes of buffer, after the call.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getHostname(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Get the MAC address of the Ethernet interface.
    /// 
    /// \param buffer alias to an array of uint8_t[6]
    /// \param bufferLength length of buffer that's writeable, should be > 6.
    /// \param unloadedLength Length of occupied bytes of buffer, after the call.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getMACAddress(uint8_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Set the port of a TCPIP service on the device.
    /// 
    /// \param service The index of the service to set the port for.
    /// \param port The port to be used for the TCPIP server.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setInterfacePort(const uint8_t service, const uint16_t port);

    /// Get the port of a TCPIP service on the device.
    /// 
    /// \param service The index of the service to get the port for.
    /// \param port The port of the TCPIP server.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getInterfacePort(const uint8_t service, uint16_t* port);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_ETHERNET_CPP_H__