/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_PoEClass_CPP.h                                    //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2026 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_POE_CPP_H__
#define __AUTOGEN_POE_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// PoEClass:
/// This entity is only available on certain modules, and provides a Power over Ethernet
///   control ability.
/// 
class aLIBEXPORT PoEClass : public EntityClass {

public:

    /// Constructor.
    PoEClass(void);

    /// Destructor.
    virtual ~PoEClass(void);

    /// Initialize the PoE Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the PoE entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Gets the current enable value of the indicated POE pair.
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param enable 1 = Enabled; 0 = Disabled;
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPairEnabled(const uint8_t pair, uint8_t* enable);

    /// Enables or disables the indicated POE pair.
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param enable 1 = Enable port; 0 = Disable port.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPairEnabled(const uint8_t pair, const uint8_t enable);

    /// Gets the power mode of the device
    /// 
    /// \param value The power mode (PD, PSE, Auto, Off).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerMode(uint8_t* value);

    /// Sets the power mode of the device
    /// 
    /// \param value The power mode (PD, PSE, Auto, Off).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerMode(const uint8_t value);

    /// Gets the power state of the device
    /// 
    /// \param value The power state (PD, PSE, Off).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerState(uint8_t* value);

    /// Gets the sourcing class for a given pair.
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param value The POE class being offered by the device (PSE).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPairSourcingClass(const uint8_t pair, uint8_t* value);

    /// Sets the sourcing class for a given pair.
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param value The POE class being offered by the device (PSE).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPairSourcingClass(const uint8_t pair, const uint8_t value);

    /// Gets the requested class for a given pair.
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param value The requested POE class by the device (PD).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPairRequestedClass(const uint8_t pair, uint8_t* value);

    /// Gets the discovered class for a given pair.
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param value The negotiated POE class by the device (PSE/PD).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPairDiscoveredClass(const uint8_t pair, uint8_t* value);

    /// Gets detected status of the POE connection for a given pair.
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param value The current detected status of the pairs.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPairDetectionStatus(const uint8_t pair, uint8_t* value);

    /// Gets the Voltage for a given pair.
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param microvolts The voltage in microvolts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPairVoltage(const uint8_t pair, int32_t* microvolts);

    /// Gets the Current for a given pair.
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param microamps The current in microamps (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPairCurrent(const uint8_t pair, int32_t* microamps);

    /// Gets the Resistance for a given pair.
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param milliohms The resistance in milliohms (1 == 1e-3Z).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPairResistance(const uint8_t pair, int32_t* milliohms);

    /// Gets the Capacitance for a given pair
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param nanofarads The capacitance in nanofarads (1 == 1e-9F).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPairCapacitance(const uint8_t pair, int32_t* nanofarads);

    /// Get the instantaneous power consumption for a given pair
    /// The equivalent of Voltage x Current
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param power Variable to be filled with the pairs power in milli-watts (mW).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPairPower(const uint8_t pair, int32_t* power);

    /// Gets the total instantaneous power consumption
    /// The equivalent of Pair1(Voltage x Current) + Pair2(Voltage x Current)
    /// 
    /// \param power Variable to be filled with the total POE power in milli-watts (mW).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getTotalPower(int32_t* power);

    /// Gets the accumulated power for a given pair.
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param power Variable to be filled with the total accumulated POE power in milli-watts
    ///        (mW).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPairAccumulatedPower(const uint8_t pair, int32_t* power);

    /// Sets the accumulated power for a given pair.
    /// 
    /// \param pair Selects PoE pair to access
    ///            - 0 = Pair 1/2
    ///            - 1 = Pair 3/4
    /// \param power The power accumulator value to be set in milli-watts (mW).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPairAccumulatedPower(const uint8_t pair, const int32_t power);

    /// Gets the total Accumulated Power
    /// 
    /// \param power Variable to be filled with the total accumulated POE power in milli-watts
    ///        (mW).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getTotalAccumulatedPower(int32_t* power);

    /// Sets the total accumulated power
    /// 
    /// \param power The power accumulator value to be set in milli-watts (mW).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setTotalAccumulatedPower(const int32_t power);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_POE_CPP_H__