/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_PowerDeliveryClass_CPP.h                          //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2026 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_POWERDELIVERY_CPP_H__
#define __AUTOGEN_POWERDELIVERY_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// PowerDeliveryClass:
/// Power Delivery or PD is a power specification which allows more charging options and
///   device behaviors within the USB interface.
/// This Entity will allow you to directly access the vast landscape of PD.
/// 
class aLIBEXPORT PowerDeliveryClass : public EntityClass {

public:

    /// Constructor.
    PowerDeliveryClass(void);

    /// Destructor.
    virtual ~PowerDeliveryClass(void);

    /// Initialize the PowerDelivery Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the PowerDelivery entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Gets the current state of the connection in the form of an enumeration.
    /// 
    /// \param state Pointer to be filled with the current connection state.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getConnectionState(uint8_t* state);

    /// Gets the number of Power Data Objects (PDOs) for a given partner and power role.
    /// 
    /// \param partner Indicates which side of the PD connection is in question.
    ///            - Local = 0 = powerdeliveryPartnerLocal
    ///            - Remote = 1 = powerdeliveryPartnerRemote
    /// \param powerRole Indicates which power role of PD connection is in question.
    ///            - Source = 1 = powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    /// \param numRules Variable to be filled with the number of PDOs.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getNumberOfPowerDataObjects(const uint8_t partner, const uint8_t powerRole, uint8_t* numRules);

    /// Gets the Power Data Object (PDO) for the requested partner, powerRole and index.
    /// 
    /// \param partner Indicates which side of the PD connection is in question.
    ///            - Local = 0 = powerdeliveryPartnerLocal
    ///            - Remote = 1 = powerdeliveryPartnerRemote
    /// \param powerRole Indicates which power role of PD connection is in question.
    ///            - Source = 1 = powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    /// \param ruleIndex The index of the PDO in question. Valid index are 1-7.
    /// \param pdo Variable to be filled with the requested power rule.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerDataObject(const uint8_t partner, const uint8_t powerRole, const uint8_t ruleIndex, uint32_t* pdo);

    /// Sets the Power Data Object (PDO) of the local partner for a given power role and index.
    /// 
    /// \param powerRole Indicates which power role of PD connection is in question.
    ///            - Source = 1 = powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    /// \param ruleIndex The index of the PDO in question. Valid index are 1-7.
    /// \param pdo Power Data Object to be set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerDataObject(const uint8_t powerRole, const uint8_t ruleIndex, const uint32_t pdo);

    /// Resets the Power Data Object (PDO) of the Local partner for a given power role and index.
    /// 
    /// \param powerRole Indicates which power role of PD connection is in question.
    ///            - Source = 1 = powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    /// \param ruleIndex The index of the PDO in question. Valid index are 1-7.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr resetPowerDataObjectToDefault(const uint8_t powerRole, const uint8_t ruleIndex);

    /// Gets all Power Data Objects (PDOs).
    /// Equivalent to calling PowerDeliveryClass::getPowerDataObject() on all partners, power
    ///   roles, and index's.
    /// 
    /// \param buffer pointer to the start of a c style buffer to be filled
    ///        The order of which is:
    ///            - Rules 1-7 Local Source
    ///            - Rules 1-7 Local Sink
    ///            - Rules 1-7 Partner Source
    ///            - Rules 1-7 Partner Sink.
    /// \param bufferLength Length of the buffer to be filed
    /// \param unloadedLength Length that was actually received and filled.
    ///        On success this value should be 28 (7 rules * 2 partners * 2 power roles)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerDataObjectList(uint32_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Gets the enabled state of the Local Power Data Object (PDO) for a given power role and
    ///   index.
    /// Enabled refers to whether the PDO will be advertised when a PD connection is made.
    /// This does not indicate the currently active rule index. This information can be found in
    ///   Request Data Object (RDO).
    /// 
    /// \param powerRole Indicates which power role of PD connection is in question.
    ///            - Source = 1 = powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    /// \param ruleIndex The index of the PDO in question. Valid index are 1-7.
    /// \param enabled Variable to be filled with enabled state.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerDataObjectEnabled(const uint8_t powerRole, const uint8_t ruleIndex, uint8_t* enabled);

    /// Sets the enabled state of the Local Power Data Object (PDO) for a given powerRole and
    ///   index.
    /// Enabled refers to whether the PDO will be advertised when a PD connection is made.
    /// This does not indicate the currently active rule index. This information can be found in
    ///   Request Data Object (RDO).
    /// 
    /// \param powerRole Indicates which power role of PD connection is in question.
    ///            - Source = 1 = powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    /// \param ruleIndex The index of the PDO in question. Valid index are 1-7.
    /// \param enabled The state to be set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerDataObjectEnabled(const uint8_t powerRole, const uint8_t ruleIndex, const uint8_t enabled);

    /// Gets all Power Data Object enables for a given power role.
    /// Equivalent of calling PowerDeliveryClass::getPowerDataObjectEnabled() for all indexes.
    /// 
    /// \param powerRole Indicates which power role of PD connection is in question.
    ///            - Source = 1 = powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    /// \param enabledList Variable to be filled with a mapped representation of the enabled PDOs
    ///        for a given power role.
    ///        Values align with a given rule index (bits 1-7, bit 0 is invalid)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerDataObjectEnabledList(const uint8_t powerRole, uint8_t* enabledList);

    /// Gets the current Request Data Object (RDO) for a given partner.
    /// RDOs are provided by the sinking device and exist only after a successful PD negotiation
    ///   (Otherwise zero).
    /// Only one RDO can exist at a time. i.e. Either the Local or Remote partner RDO
    /// 
    /// \param partner Indicates which side of the PD connection is in question.
    ///            - Local = 0 = powerdeliveryPartnerLocal
    ///            - Remote = 1 = powerdeliveryPartnerRemote
    /// \param rdo Variable to be filled with the current RDO.
    ///        Zero indicates the RDO is not active.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getRequestDataObject(const uint8_t partner, uint32_t* rdo);

    /// Sets the current Request Data Object (RDO) for a given partner.
    /// Only the local partner can be changed.
    /// RDOs are provided by the sinking device and exist only after a successful PD negotiation
    ///   (Otherwise zero).
    /// Only one RDO can exist at a time. i.e. Either the Local or Remote partner RDO
    /// 
    /// \param rdo Request Data Object to be set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setRequestDataObject(const uint32_t rdo);

    /// Gets the current state of the connection in the form of a bitmask.
    /// 
    /// \param state Pointer to be filled with the current connection state bits.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getLinkState(uint32_t* state);

    /// Gets the length of time that the port has been in the attached state.
    /// Returned as a list of two unsigned integers, first seconds, then microseconds.
    /// 
    /// \param secondsAndMicroseconds Pointer to list of unsigned integers to fill with attach
    ///        time elapsed
    /// \param bufferLength Length of the buffer to be filed
    /// \param unloadedLength Length that was actually received and filled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getAttachTimeElapsed(uint32_t* secondsAndMicroseconds, const size_t bufferLength, size_t* unloadedLength);

    /// Gets the power roles that may be advertised by the local partner. (CC Strapping).
    /// 
    /// \param powerRole Variable to be filed with the power role
    ///            - None = 0 = pdPowerRoleCapabilities_None
    ///            - Source = 1 = pdPowerRoleCapabilities_Source
    ///            - Sink = 2 = pdPowerRoleCapabilities_Sink
    ///            - Source/Sink = 3 = pdPowerRoleCapabilities_DualRole (Dual Role Port)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerRoleCapabilities(uint8_t* powerRole);

    /// Gets the power role that is currently being advertised by the local partner. (CC
    ///   Strapping).
    /// 
    /// \param powerRole Variable to be filed with the power role
    ///            - Disabled = 0 = powerdeliveryPowerRoleDisabled
    ///            - Source = 1= powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    ///            - Source/Sink = 3 = powerdeliveryPowerRoleSourceSink (Dual Role Port)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerRole(uint8_t* powerRole);

    /// Set the current power role to be advertised by the Local partner. (CC Strapping).
    /// 
    /// \param powerRole Value to be applied.
    ///            - Disabled = 0 = powerdeliveryPowerRoleDisabled
    ///            - Source = 1= powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    ///            - Source/Sink = 3 = powerdeliveryPowerRoleSourceSink (Dual Role Port)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerRole(const uint8_t powerRole);

    /// Gets the preferred power role currently being advertised by the Local partner. (CC
    ///   Strapping).
    /// 
    /// \param powerRole Value to be applied.
    ///            - Disabled = 0 = powerdeliveryPowerRoleDisabled
    ///            - Source = 1= powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerRolePreferred(uint8_t* powerRole);

    /// Set the preferred power role to be advertised by the Local partner (CC Strapping).
    /// 
    /// \param powerRole Value to be applied.
    ///            - Disabled = 0 = powerdeliveryPowerRoleDisabled
    ///            - Source = 1= powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerRolePreferred(const uint8_t powerRole);

    /// Gets the data roles that may be advertised by the local partner.
    /// 
    /// \param dataRole Variable to be filed with the data role
    ///            - None = 0 = pdDataRoleCapabilities_None
    ///            - DFP = 1 = pdDataRoleCapabilities_DFP
    ///            - UFP = 2 = pdDataRoleCapabilities_UFP
    ///            - DFP/UFP = 3 = pdDataRoleCapabilities_DualRole (Dual Role Port)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataRoleCapabilities(uint8_t* dataRole);

    /// Gets the maximum voltage capability reported by the e-mark of the attached cable.
    /// 
    /// \param maxVoltage Variable to be filled with an enumerated representation of voltage.
    ///            - Unknown/Unattached (0)
    ///            - 20 Volts DC (1)
    ///            - 30 Volts DC (2)
    ///            - 40 Volts DC (3)
    ///            - 50 Volts DC (4)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCableVoltageMax(uint8_t* maxVoltage);

    /// Gets the maximum current capability report by the e-mark of the attached cable.
    /// 
    /// \param maxCurrent Variable to be filled with an enumerated representation of current.
    ///            - Unknown/Unattached (0)
    ///            - 3 Amps (1)
    ///            - 5 Amps (2)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCableCurrentMax(uint8_t* maxCurrent);

    /// Gets the maximum data rate capability reported by the e-mark of the attached cable.
    /// 
    /// \param maxSpeed Variable to be filled with an enumerated representation of data speed.
    ///            - Unknown/Unattached (0)
    ///            - USB 2.0 (1)
    ///            - USB 3.2 gen 1 (2)
    ///            - USB 3.2 / USB 4 gen 2 (3)
    ///            - USB 4 gen 3 (4)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCableSpeedMax(uint8_t* maxSpeed);

    /// Gets the cable type reported by the e-mark of the attached cable.
    /// 
    /// \param type Variable to be filled with an enumerated representation of the cable type.
    ///            - Invalid, no e-mark and not Vconn powered (0)
    ///            - Passive cable with e-mark (1)
    ///            - Active cable (2)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCableType(uint8_t* type);

    /// Gets the current orientation being used for PD communication
    /// 
    /// \param orientation Variable filled with an enumeration of the orientation.
    ///            - Unconnected (0)
    ///            - CC1 (1)
    ///            - CC2 (2)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCableOrientation(uint8_t* orientation);

    /// Requests an action of the Remote partner.
    /// Actions are not guaranteed to occur.
    /// 
    /// \param request Request to be issued to the remote partner
    ///            - pdRequestHardReset (0)
    ///            - pdRequestSoftReset (1)
    ///            - pdRequestDataReset (2)
    ///            - pdRequestPowerRoleSwap (3)
    ///            - pdRequestPowerFastRoleSwap (4)
    ///            - pdRequestDataRoleSwap (5)
    ///            - pdRequestVconnSwap (6)
    ///            - pdRequestSinkGoToMinimum (7)
    ///            - pdRequestRemoteSourcePowerDataObjects (8)
    ///            - pdRequestRemoteSinkPowerDataObjects (9)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr request(const uint8_t request);

    /// Gets the status of the last request command sent.
    /// 
    /// \param status Variable to be filled with the status
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr requestStatus(uint32_t* status);

    /// Gets the current enabled overrides
    /// 
    /// \param overrides Bit mapped representation of the current override configuration.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getOverride(uint32_t* overrides);

    /// Sets the current enabled overrides
    /// 
    /// \param overrides Overrides to be set in a bit mapped representation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setOverride(const uint32_t overrides);

    /// Gets the current mode of the local partner flag/advertisement.
    /// These flags are apart of the first Local Power Data Object and must be managed in order to
    ///   accurately represent the system to other PD devices.
    /// This API allows overriding of that feature. Overriding may lead to unexpected behaviors.
    /// 
    /// \param flag Flag/Advertisement to be modified
    /// \param mode Variable to be filled with the current mode.
    ///            - Disabled (0)
    ///            - Enabled (1)
    ///            - Auto (2) default
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getFlagMode(const uint8_t flag, uint8_t* mode);

    /// Sets how the local partner flag/advertisement is managed.
    /// These flags are apart of the first Local Power Data Object and must be managed in order to
    ///   accurately represent the system to other PD devices.
    /// This API allows overriding of that feature. Overriding may lead to unexpected behaviors.
    /// 
    /// \param flag Flag/Advertisement to be modified
    /// \param mode Value to be applied.
    ///            - Disabled (0)
    ///            - Enabled (1)
    ///            - Auto (2) default
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setFlagMode(const uint8_t flag, const uint8_t mode);

    /// Gets the Peak Current Configuration for the Local Source.
    /// The peak current configuration refers to the allowable tolerance/overload capabilities in
    ///   regards to the devices max current.
    /// This tolerance includes a maximum value and a time unit.
    /// 
    /// \param configuration An enumerated value referring to the current configuration.
    ///            - Allowable values are 0 - 4
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPeakCurrentConfiguration(uint8_t* configuration);

    /// Sets the Peak Current Configuration for the Local Source.
    /// The peak current configuration refers to the allowable tolerance/overload capabilities in
    ///   regards to the devices max current.
    /// This tolerance includes a maximum value and a time unit.
    /// 
    /// \param configuration An enumerated value referring to the configuration to be set
    ///            - Allowable values are 0 - 4
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPeakCurrentConfiguration(const uint8_t configuration);

    /// Gets the Fast Role Swap Current
    /// The fast role swap current refers to the amount of current required by the Local Sink in
    ///   order to successfully preform the swap.
    /// 
    /// \param swapCurrent An enumerated value referring to current swap value.
    ///            - 0A (0)
    ///            - 900mA (1)
    ///            - 1.5A (2)
    ///            - 3A (3)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getFastRoleSwapCurrent(uint8_t* swapCurrent);

    /// Sets the Fast Role Swap Current
    /// The fast role swap current refers to the amount of current required by the Local Sink in
    ///   order to successfully preform the swap.
    /// 
    /// \param swapCurrent An enumerated value referring to value to be set.
    ///            - 0A (0)
    ///            - 900mA (1)
    ///            - 1.5A (2)
    ///            - 3A (3)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setFastRoleSwapCurrent(const uint8_t swapCurrent);

    /// Helper function for packing Data Object attributes.
    /// This value is used as a subindex for all Data Object calls with the BrainStem Protocol.
    /// 
    /// \param attributes Variable to be filled with packed values.
    /// \param partner Indicates which side of the PD connection.
    ///            - Local = 0 = powerdeliveryPartnerLocal
    ///            - Remote = 1 = powerdeliveryPartnerRemote
    /// \param powerRole Indicates which power role of PD connection.
    ///            - Source = 1 = powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    /// \param ruleIndex Data object index.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
static     aErr packDataObjectAttributes(uint8_t* attributes, const uint8_t partner, const uint8_t powerRole, const uint8_t ruleIndex);

    /// Helper function for unpacking Data Object attributes.
    /// This value is used as a subindex for all Data Object calls with the BrainStem Protocol.
    /// 
    /// \param attributes Variable to be filled with packed values.
    /// \param partner Indicates which side of the PD connection.
    ///            - Local = 0 = powerdeliveryPartnerLocal
    ///            - Remote = 1 = powerdeliveryPartnerRemote
    /// \param powerRole Indicates which power role of PD connection.
    ///            - Source = 1 = powerdeliveryPowerRoleSource
    ///            - Sink = 2 = powerdeliveryPowerRoleSink
    /// \param ruleIndex Data object index.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
static     aErr unpackDataObjectAttributes(const uint8_t attributes, uint8_t* partner, uint8_t* powerRole, uint8_t* ruleIndex);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_POWERDELIVERY_CPP_H__