/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_StoreClass_CPP.h                                  //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2026 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_STORE_CPP_H__
#define __AUTOGEN_STORE_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// StoreClass:
/// The store provides a flat file system on modules that have storage capacity.
/// Files are referred to as slots and they have simple zero-based numbers for access.
/// Store slots can be used for generalized storage and commonly contain compiled reflex code
///   (files ending in .map) or templates used by the system.
/// Slots simply contain bytes with no expected organization but the code or use of the slot
///   may impose a structure.
/// Stores have fixed indices based on type.
/// Not every module contains a store of each type. Consult the module datasheet for details
///   on which specific stores are implemented, if any, and the capacities of implemented
///   stores.
/// 
class aLIBEXPORT StoreClass : public EntityClass {

public:

    /// Constructor.
    StoreClass(void);

    /// Destructor.
    virtual ~StoreClass(void);

    /// Initialize the Store Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the Store entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Get slot state.
    /// 
    /// \param slot The slot number.
    /// \param state true: enabled, false: disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSlotState(const uint8_t slot, uint8_t* state);

    /// Load the slot.
    /// 
    /// \param slot The slot number.
    /// \param buffer The data.
    /// \param bufferLength The data length.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr loadSlot(const uint8_t slot, const uint8_t* buffer, const uint16_t bufferLength);

    /// Unload the slot data.
    /// 
    /// \param slot The slot number.
    /// \param bufferLength The length of buffer buffer in bytes. This is the maximum number of
    ///        bytes that should be unloaded.
    /// \param buffer Byte array that the unloaded data will be placed into.
    /// \param unloadedLength Length of data that was unloaded. Unloaded length
    ///                              will never be larger than dataLength.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr unloadSlot(const uint8_t slot, const size_t bufferLength, uint8_t* buffer, size_t* unloadedLength);

    /// Enable slot.
    /// 
    /// \param slot The slot number.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr slotEnable(const uint8_t slot);

    /// Disable slot.
    /// 
    /// \param slot The slot number.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr slotDisable(const uint8_t slot);

    /// Get the slot capacity.
    /// Returns the Capacity of the slot, i.e. The number of bytes it can hold.
    /// 
    /// \param slot The slot number.
    /// \param capacity The slot capacity.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSlotCapacity(const uint8_t slot, size_t* capacity);

    /// Get the slot size.
    /// The slot size represents the size of the data currently filling the slot in bytes.
    /// 
    /// \param slot The slot number.
    /// \param size The slot size.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSlotSize(const uint8_t slot, size_t* size);

    /// Gets the current lock state of the slot
    /// Allows for write protection on a slot.
    /// 
    /// \param slot The slot number
    /// \param lock Variable to be filed with the locked state.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSlotLocked(const uint8_t slot, uint8_t* lock);

    /// Sets the locked state of the slot
    /// Allows for write protection on a slot.
    /// 
    /// \param slot The slot number
    /// \param lock state to be set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setSlotLocked(const uint8_t slot, const uint8_t lock);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_STORE_CPP_H__