/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_UARTClass_CPP.h                                   //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2026 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_UART_CPP_H__
#define __AUTOGEN_UART_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// UARTClass:
/// A UART is a "Universal Asynchronous Receiver/Transmitter.  Many times referred to as a COM
///   (communication), Serial, or TTY (teletypewriter) port.
/// The UART Class allows the enabling and disabling of the UART data lines.
/// 
class aLIBEXPORT UARTClass : public EntityClass {

public:

    /// Constructor.
    UARTClass(void);

    /// Destructor.
    virtual ~UARTClass(void);

    /// Initialize the UART Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the UART entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Enable the UART channel.
    /// 
    /// \param enabled true: enabled, false: disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setEnable(const uint8_t enabled);

    /// Get the enabled state of the uart.
    /// 
    /// \param enabled true: enabled, false: disabled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getEnable(uint8_t* enabled);

    /// Set the UART baud rate.
    /// If zero, automatic baud rate selection is used.
    /// 
    /// \param rate baud rate.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setBaudRate(const uint32_t rate);

    /// Get the UART baud rate.
    /// If zero, automatic baud rate selection is used.
    /// 
    /// \param rate Pointer variable to be filled with baud rate.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getBaudRate(uint32_t* rate);

    /// Set the UART protocol.
    /// 
    /// \param protocol An enumeration of serial protocols.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setProtocol(const uint8_t protocol);

    /// Get the UART protocol.
    /// 
    /// \param protocol Pointer to where result is placed.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getProtocol(uint8_t* protocol);

    /// Set the index of another UART Entity that should be linked to this UART.
    /// 
    /// If set to the index of this entity, the channel will not be linked.
    /// If set to the index of another UART entity, data will be sent between the two UART
    ///   entities with no additional processing.
    /// 
    /// \param channel Index of the UART Entity to link
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setLinkChannel(const uint8_t channel);

    /// Gets the index of the UART Entity that this entity is linked to.
    /// 
    /// \param channel Pointer to where result is placed.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getLinkChannel(uint8_t* channel);

    /// Set the UART stop bit configuration
    /// 
    /// \param stopBits Stop Bits of UART Channel. Allowed options:
    ///            - uartStopBits_1_Value
    ///            - uartStopBits_1p5_Value
    ///            - uartStopBits_2_Value
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setStopBits(const uint8_t stopBits);

    /// Set the UART stop bit configuration
    /// 
    /// \param stopBits Pointer to where result is placed. Possible values:
    ///            - uartStopBits_1_Value
    ///            - uartStopBits_1p5_Value
    ///            - uartStopBits_2_Value
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getStopBits(uint8_t* stopBits);

    /// Set the UART parity.
    /// 
    /// \param parity Parity of UART Channel. Allowed options:
    ///            - uartParity_None_Value
    ///            - uartParity_Odd_Value
    ///            - uartParity_Even_Value
    ///            - uartParity_Mark_Value
    ///            - uartParity_Space_Value
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setParity(const uint8_t parity);

    /// Get the UART parity.
    /// 
    /// \param parity Pointer variable to be filled with value. Possible values:
    ///            - uartParity_None_Value
    ///            - uartParity_Odd_Value
    ///            - uartParity_Even_Value
    ///            - uartParity_Mark_Value
    ///            - uartParity_Space_Value
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getParity(uint8_t* parity);

    /// Set the number of bits per character
    /// 
    /// \param dataBits Data Bits of UART Channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataBits(const uint8_t dataBits);

    /// Get the number of bits per character
    /// 
    /// \param dataBits Pointer to where result is placed.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataBits(uint8_t* dataBits);

    /// Set the UART flow control configuration
    /// 
    /// \param flowControl Flow Control of UART Channel as a bitmask. Allowed bits:
    ///            - uartFlowControl_RTS_CTS_Bit
    ///            - uartFlowControl_DSR_DTR_Bit
    ///            - uartFlowControl_XON_XOFF_Bit
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setFlowControl(const uint8_t flowControl);

    /// Set the UART flow control configuration
    /// 
    /// \param flowControl Pointer to bitmask where result is placed. Possible bits:
    ///            - uartFlowControl_RTS_CTS_Bit
    ///            - uartFlowControl_DSR_DTR_Bit
    ///            - uartFlowControl_XON_XOFF_Bit
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getFlowControl(uint8_t* flowControl);

    /// Returns a bitmask containing a list of protocols that this UART entity is allowed to
    ///   select.
    /// This does not guarantee that selecting a protocol with "setProtocol" will have an
    ///   available resource.
    /// 
    /// \param protocols Bitmask containing list of protocols that may be selected.
    ///        The value of the uartProtocol is mapped to the bit index (e.g.
    ///        uartProtocol_Undefined is bit 0, uartProtocol_ExtronResponder_Value is bit 1, etc.)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCapableProtocols(uint32_t* protocols);

    /// Returns a bitmask containing a list of protocols that this UART entity is capable of
    ///   selecting, and has an available protocol resource to assign.
    /// 
    /// \param protocols Bitmask containing list of protocols that are available to select.
    ///        The value of the uartProtocol is mapped to the bit index (e.g.
    ///        uartProtocol_Undefined is bit 0, uartProtocol_ExtronResponder_Value is bit 1, etc.)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getAvailableProtocols(uint32_t* protocols);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_UART_CPP_H__