/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_USBClass_CPP.h                                    //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2026 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_USB_CPP_H__
#define __AUTOGEN_USB_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// USBClass:
/// The USB class provides methods to interact with a USB hub and USB switches.
/// Different USB hub products have varying support; check the datasheet to understand the
///   capabilities of each product.
/// 
class aLIBEXPORT USBClass : public EntityClass {

public:

    /// Constructor.
    USBClass(void);

    /// Destructor.
    virtual ~USBClass(void);

    /// Initialize the USB Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the USB entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Enable both power and data lines for a port.
    /// 
    /// \param channel The USB sub channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPortEnable(const uint8_t channel);

    /// Disable both power and data lines for a port.
    /// 
    /// \param channel The USB sub channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPortDisable(const uint8_t channel);

    /// Enable the only the data lines for a port without changing the state of the power line.
    /// 
    /// \param channel The USB sub channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataEnable(const uint8_t channel);

    /// Disable only the data lines for a port without changing the state of the power line.
    /// 
    /// \param channel The USB sub channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataDisable(const uint8_t channel);

    /// Enable the only the data lines for a port without changing the state of the power line,
    ///   Hi-Speed (2.0) only.
    /// 
    /// \param channel The USB sub channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setHiSpeedDataEnable(const uint8_t channel);

    /// Disable only the data lines for a port without changing the state of the power line, Hi-
    ///   Speed (2.0) only.
    /// 
    /// \param channel The USB sub channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setHiSpeedDataDisable(const uint8_t channel);

    /// Enable the only the data lines for a port without changing the state of the power line,
    ///   SuperSpeed (3.0) only.
    /// 
    /// \param channel The USB sub channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setSuperSpeedDataEnable(const uint8_t channel);

    /// Disable only the data lines for a port without changing the state of the power line,
    ///   SuperSpeed (3.0) only.
    /// 
    /// \param channel The USB sub channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setSuperSpeedDataDisable(const uint8_t channel);

    /// Enable only the power line for a port without changing the state of the data lines.
    /// 
    /// \param channel The USB sub channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerEnable(const uint8_t channel);

    /// Disable only the power line for a port without changing the state of the data lines.
    /// 
    /// \param channel The USB sub channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerDisable(const uint8_t channel);

    /// Get the current through the power line for a port.
    /// 
    /// \param channel The USB sub channel.
    /// \param microamps The USB channel current in micro-amps (1 == 1e-6A).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPortCurrent(const uint8_t channel, int32_t* microamps);

    /// Get the voltage on the power line for a port.
    /// 
    /// \param channel The USB sub channel.
    /// \param microvolts The USB channel voltage in microvolts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPortVoltage(const uint8_t channel, int32_t* microvolts);

    /// Get a bit mapped representation of the hubs mode; see the product datasheet for mode
    ///   mapping and meaning.
    /// 
    /// \param mode The USB hub mode.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getHubMode(uint32_t* mode);

    /// Set a bit mapped hub state; see the product datasheet for state mapping and meaning.
    /// 
    /// \param mode The USB hub mode.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setHubMode(const uint32_t mode);

    /// Clear the error status for the given port.
    /// 
    /// \param channel The port to clear error status for.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr clearPortErrorStatus(const uint8_t channel);

    /// Get the upstream switch mode for the USB upstream ports. Returns auto, port 0 or port 1.
    /// 
    /// \param mode The Upstream port mode.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getUpstreamMode(uint8_t* mode);

    /// Set the upstream switch mode for the USB upstream ports. Values are usbUpstreamModeAuto,
    ///   usbUpstreamModePort0, usbUpstreamModePort1, and usbUpstreamModeNone.
    /// 
    /// \param mode The Upstream port mode.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setUpstreamMode(const uint8_t mode);

    /// Get the upstream switch state for the USB upstream ports.
    /// Returns 2 if no ports plugged in, 0 if the mode is set correctly and a cable is plugged
    ///   into port 0, and 1 if the mode is set correctly and a cable is plugged into port 1.
    /// 
    /// \param state The Upstream port state.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getUpstreamState(uint8_t* state);

    /// Set the inter-port enumeration delay in milliseconds.
    /// 
    /// \param ms_delay Millisecond delay in 100mS increments (100, 200, 300 etc.)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setEnumerationDelay(const uint32_t ms_delay);

    /// Get the inter-port enumeration delay in milliseconds.
    /// 
    /// \param ms_delay Millisecond delay in 100mS increments (100, 200, 300 etc.)
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getEnumerationDelay(uint32_t* ms_delay);

    /// Set the current limit for the port.
    /// If the set limit is not achievable, devices will round down to the nearest available
    ///   current limit setting.
    /// This setting can be saved with a stem.system.save() call.
    /// 
    /// \param channel USB downstream channel to limit.
    /// \param microamps The current limit setting.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPortCurrentLimit(const uint8_t channel, const uint32_t microamps);

    /// Get the current limit for the port.
    /// 
    /// \param channel USB downstream channel to limit.
    /// \param microamps The current limit setting.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPortCurrentLimit(const uint8_t channel, uint32_t* microamps);

    /// Set the mode for the Port.
    /// The mode is a bitmapped representation of the capabilities of the usb port.
    /// These capabilities change for each of the BrainStem devices which implement the usb
    ///   entity.
    /// See your device reference page for a complete list of capabilities.
    /// Some devices use a common bit mapping for port mode, as sub-definitions of usbPortMode.
    /// 
    /// \param channel USB downstream channel to set the mode on.
    /// \param mode The port mode setting as packed bit field.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPortMode(const uint8_t channel, const uint32_t mode);

    /// Get the current mode for the Port.
    /// The mode is a bitmapped representation of the capabilities of the usb port.
    /// These capabilities change for each of the BrainStem devices which implement the usb
    ///   entity.
    /// See your device reference page for a complete list of capabilities.
    /// Some devices use a common bit mapping for port mode, as sub-definitions of usbPortMode.
    /// 
    /// \param channel USB downstream channel.
    /// \param mode The port mode setting.
    ///        Mode will be filled with the current setting.
    ///        Mode bits that are not used will be marked as don't care.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPortMode(const uint8_t channel, uint32_t* mode);

    /// Get the current State for the Port.
    /// 
    /// \param channel USB downstream channel.
    /// \param state The port mode setting.
    ///        Mode will be filled with the current setting.
    ///        Mode bits that are not used will be marked as don't care.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPortState(const uint8_t channel, uint32_t* state);

    /// Get the current error for the Port.
    /// 
    /// \param channel USB downstream channel.
    /// \param error The port mode setting.
    ///        Mode will be filled with the current setting.
    ///        Mode bits that are not used will be marked as don't care.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPortError(const uint8_t channel, uint32_t* error);

    /// Set the upstream boost mode.
    /// Boost mode increases the drive strength of the USB data signals (power signals are not
    ///   changed).
    /// Boosting the data signal strength may help to overcome connectivity issues when using long
    ///   cables or connecting through "pogo" pins.
    /// Possible modes are 0 - no boost, 1 - 4% boost, 2 - 8% boost, 3 - 12% boost.
    /// This setting is not applied until a stem.system.save() call and power cycle of the hub.
    /// Setting is then persistent until changed or the hub is reset.
    /// After reset, default value of 0% boost is restored.
    /// 
    /// \param setting Upstream boost setting 0, 1, 2, or 3.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setUpstreamBoostMode(const uint8_t setting);

    /// Set the downstream boost mode.
    /// Boost mode increases the drive strength of the USB data signals (power signals are not
    ///   changed).
    /// Boosting the data signal strength may help to overcome connectivity issues when using long
    ///   cables or connecting through "pogo" pins.
    /// Possible modes are 0 - no boost, 1 - 4% boost, 2 - 8% boost, 3 - 12% boost.
    /// This setting is not applied until a stem.system.save() call and power cycle of the hub.
    /// Setting is then persistent until changed or the hub is reset.
    /// After reset, default value of 0% boost is restored.
    /// 
    /// \param setting Downstream boost setting 0, 1, 2, or 3.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDownstreamBoostMode(const uint8_t setting);

    /// Get the upstream boost mode.
    /// Possible modes are 0 - no boost, 1 - 4% boost, 2 - 8% boost, 3 - 12% boost.
    /// 
    /// \param setting The current Upstream boost setting 0, 1, 2, or 3.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getUpstreamBoostMode(uint8_t* setting);

    /// Get the downstream boost mode.
    /// Possible modes are 0 - no boost, 1 - 4% boost, 2 - 8% boost, 3 - 12% boost.
    /// 
    /// \param setting The current Downstream boost setting 0, 1, 2, or 3.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDownstreamBoostMode(uint8_t* setting);

    /// Get the current data transfer speed for the downstream port.
    /// The data speed can be Hi-Speed (2.0) or SuperSpeed (3.0) depending on what the downstream
    ///   device attached is using
    /// 
    /// \param channel USB downstream channel to check.
    /// \param speed Filled with the current port data speed
    ///            - N/A: usbDownstreamDataSpeed_na = 0
    ///            - Hi Speed: usbDownstreamDataSpeed_hs = 1
    ///            - SuperSpeed: usbDownstreamDataSpeed_ss = 2
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDownstreamDataSpeed(const uint8_t channel, uint8_t* speed);

    /// Sets the connect mode of the switch.
    /// 
    /// \param channel The USB sub channel.
    /// \param mode The connect mode
    ///            - usbManualConnect = 0
    ///            - usbAutoConnect = 1
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setConnectMode(const uint8_t channel, const uint8_t mode);

    /// Gets the connect mode of the switch.
    /// 
    /// \param channel The USB sub channel.
    /// \param mode The current connect mode
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getConnectMode(const uint8_t channel, uint8_t* mode);

    /// Set Enable/Disable on the CC1 line.
    /// 
    /// \param channel USB channel.
    /// \param enable State to be set
    ///            - Disabled: 0
    ///            - Enabled: 1
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCC1Enable(const uint8_t channel, const uint8_t enable);

    /// Get Enable/Disable on the CC1 line.
    /// 
    /// \param channel USB channel.
    /// \param pEnable State to be filled
    ///            - Disabled: 0
    ///            - Enabled: 1
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC1Enable(const uint8_t channel, uint8_t* pEnable);

    /// Set Enable/Disable on the CC2 line.
    /// 
    /// \param channel USB channel.
    /// \param enable State to be filled
    ///            - Disabled: 0
    ///            - Enabled: 1
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCC2Enable(const uint8_t channel, const uint8_t enable);

    /// Get Enable/Disable on the CC2 line.
    /// 
    /// \param channel USB channel.
    /// \param pEnable State to be filled
    ///            - Disabled: 0
    ///            - Enabled: 1
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC2Enable(const uint8_t channel, uint8_t* pEnable);

    /// Get the current through the CC1 for a port.
    /// 
    /// \param channel The USB sub channel.
    /// \param microamps The USB channel current in micro-amps (1 == 1e-6A).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC1Current(const uint8_t channel, int32_t* microamps);

    /// Get the current through the CC2 for a port.
    /// 
    /// \param channel The USB sub channel.
    /// \param microamps The USB channel current in micro-amps (1 == 1e-6A).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC2Current(const uint8_t channel, int32_t* microamps);

    /// Get the voltage of CC1 for a port.
    /// 
    /// \param channel The USB sub channel.
    /// \param microvolts The USB channel voltage in micro-volts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC1Voltage(const uint8_t channel, int32_t* microvolts);

    /// Get the voltage of CC2 for a port.
    /// 
    /// \param channel The USB sub channel.
    /// \param microvolts The USB channel voltage in micro-volts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCC2Voltage(const uint8_t channel, int32_t* microvolts);

    /// Enable/Disable only the SBU1/2 based on the configuration of the usbPortMode settings.
    /// 
    /// \param channel The USB sub channel.
    /// \param enable The state to be set
    ///        - Disabled: 0
    ///        - Enabled: 1
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setSBUEnable(const uint8_t channel, const uint8_t enable);

    /// Get the Enable/Disable status of the SBU
    /// 
    /// \param channel The USB sub channel.
    /// \param pEnable The enable/disable status of the SBU
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSBUEnable(const uint8_t channel, uint8_t* pEnable);

    /// Set Cable flip. This will flip SBU, CC and SS data lines.
    /// 
    /// \param channel The USB sub channel.
    /// \param enable The state to be set The state to be set
    ///            - Disabled: 0
    ///            - Enabled: 1
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setCableFlip(const uint8_t channel, const uint8_t enable);

    /// Get Cable flip setting.
    /// 
    /// \param channel The USB sub channel.
    /// \param pEnable The enable/disable status of cable flip.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getCableFlip(const uint8_t channel, uint8_t* pEnable);

    /// Set USB Alt Mode Configuration.
    /// 
    /// \param channel The USB sub channel
    /// \param configuration The USB configuration to be set for the given channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setAltModeConfig(const uint8_t channel, const uint32_t configuration);

    /// Get USB Alt Mode Configuration.
    /// 
    /// \param channel The USB sub channel
    /// \param configuration The USB configuration for the given channel.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getAltModeConfig(const uint8_t channel, uint32_t* configuration);

    /// Get the voltage of SBU1 for a port.
    /// 
    /// \param channel The USB sub channel.
    /// \param microvolts The USB channel voltage in micro-volts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSBU1Voltage(const uint8_t channel, int32_t* microvolts);

    /// Get the voltage of SBU2 for a port.
    /// 
    /// \param channel The USB sub channel.
    /// \param microvolts The USB channel voltage in micro-volts (1 == 1e-6V).
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSBU2Voltage(const uint8_t channel, int32_t* microvolts);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_USB_CPP_H__