/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: autoGen_USBSystemClass_CPP.h                              //
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2026 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////

#ifndef __AUTOGEN_USBSYSTEM_CPP_H__
#define __AUTOGEN_USBSYSTEM_CPP_H__

#include "BrainStem-core.h"

#ifdef __GNUC__
#define DEPRECATED(...) __attribute__((deprecated(__VA_ARGS__)))
#elif defined(_MSC_VER)
#define DEPRECATED(...) __declspec(deprecated(__VA_ARGS__))
#else
#define DEPRECATED(...)
#pragma message("DEPRECATED is not defined for this compiler")
#endif

#if defined(__cplusplus)

namespace Acroname {
namespace BrainStem {

/// USBSystemClass:
/// The USBSystem class provides high level control of the lower level Port Class.
/// 
class aLIBEXPORT USBSystemClass : public EntityClass {

public:

    /// Constructor.
    USBSystemClass(void);

    /// Destructor.
    virtual ~USBSystemClass(void);

    /// Initialize the USBSystem Class.
    ///
    /// \param pModule The module to which this entity belongs.
    /// \param index The index of the USBSystem entity to be addressed.
    ///
    void init(Module* pModule, const uint8_t index);

    /// Gets the upstream port.
    /// 
    /// \param port The current upstream port.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getUpstream(uint8_t* port);

    /// Sets the upstream port.
    /// 
    /// \param port The upstream port to set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setUpstream(const uint8_t port);

    /// Gets the inter-port enumeration delay in milliseconds.
    /// Delay is applied upon hub enumeration.
    /// 
    /// \param msDelay the current inter-port delay in milliseconds.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getEnumerationDelay(uint32_t* msDelay);

    /// Sets the inter-port enumeration delay in milliseconds.
    /// Delay is applied upon hub enumeration.
    /// 
    /// \param msDelay The delay in milliseconds to be applied between port enables
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setEnumerationDelay(const uint32_t msDelay);

    /// Gets the data role of all ports with a single call
    /// Equivalent to calling PortClass::getDataRole() on each individual port.
    /// 
    /// \param roleList A bit packed representation of the data role for all ports.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataRoleList(uint32_t* roleList);

    /// Gets the current enabled status of all ports with a single call.
    /// Equivalent to calling PortClass::setEnabled() on each port.
    /// 
    /// \param enabledList Bit packed representation of the enabled status for all ports.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getEnabledList(uint32_t* enabledList);

    /// Sets the enabled status of all ports with a single call.
    /// Equivalent to calling PortClass::setEnabled() on each port.
    /// 
    /// \param enabledList Bit packed representation of the enabled status for all ports to be
    ///        applied.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setEnabledList(const uint32_t enabledList);

    /// Gets the current mode of all ports with a single call.
    /// Equivalent to calling PortClass:getMode() on each port.
    /// 
    /// \param buffer pointer to the start of a c style buffer to be filled
    /// \param bufferLength Length of the buffer to be filed
    /// \param unloadedLength Length that was actually received and filled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getModeList(uint32_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Sets the mode of all ports with a single call.
    /// Equivalent to calling PortClass::setMode() on each port
    /// 
    /// \param buffer Pointer to the start of a c style buffer to be transferred.
    /// \param bufferLength Length of the buffer to be transferred.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setModeList(const uint32_t* buffer, const size_t bufferLength);

    /// Gets the state for all ports with a single call.
    /// Equivalent to calling PortClass::getState() on each port.
    /// 
    /// \param buffer pointer to the start of a c style buffer to be filled
    /// \param bufferLength Length of the buffer to be filed
    /// \param unloadedLength Length that was actually received and filled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getStateList(uint32_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Gets the behavior of the power manager.
    /// The power manager is responsible for budgeting the power of the system, i.e. What happens
    ///   when requested power greater than available power.
    /// 
    /// \param behavior Variable to be filled with an enumerated representation of behavior.
    ///        Available behaviors are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerBehavior(uint8_t* behavior);

    /// Sets the behavior of how available power is managed, i.e. What happens when requested
    ///   power is greater than available power.
    /// 
    /// \param behavior An enumerated representation of behavior.
    ///        Available behaviors are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerBehavior(const uint8_t behavior);

    /// Gets the current power behavior configuration.
    /// Certain power behaviors use a list of ports to determine priority when budgeting power.
    /// 
    /// \param buffer pointer to the start of a c style buffer to be filled
    /// \param bufferLength Length of the buffer to be filed
    /// \param unloadedLength Length that was actually received and filled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getPowerBehaviorConfig(uint32_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Sets the current power behavior configuration.
    /// Certain power behaviors use a list of ports to determine priority when budgeting power.
    /// 
    /// \param buffer Pointer to the start of a c style buffer to be transferred.
    /// \param bufferLength Length of the buffer to be transferred.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setPowerBehaviorConfig(const uint32_t* buffer, const size_t bufferLength);

    /// Gets the behavior of how upstream and downstream ports are determined, i.e. How do you
    ///   manage requests for data role swaps and new upstream connections.
    /// 
    /// \param behavior Variable to be filled with an enumerated representation of behavior.
    ///        Available behaviors are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataRoleBehavior(uint8_t* behavior);

    /// Sets the behavior of how upstream and downstream ports are determined, i.e. How do you
    ///   manage requests for data role swaps and new upstream connections.
    /// 
    /// \param behavior An enumerated representation of behavior.
    ///        Available behaviors are product specific. See the reference documentation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataRoleBehavior(const uint8_t behavior);

    /// Gets the current data role behavior configuration.
    /// Certain data role behaviors use a list of ports to determine priority host priority.
    /// 
    /// \param buffer pointer to the start of a c style buffer to be filled
    /// \param bufferLength Length of the buffer to be filed
    /// \param unloadedLength Length that was actually received and filled.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataRoleBehaviorConfig(uint32_t* buffer, const size_t bufferLength, size_t* unloadedLength);

    /// Sets the current data role behavior configuration.
    /// Certain data role behaviors use a list of ports to determine host priority.
    /// 
    /// \param buffer Pointer to the start of a c style buffer to be transferred.
    /// \param bufferLength Length of the buffer to be transferred.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataRoleBehaviorConfig(const uint32_t* buffer, const size_t bufferLength);

    /// Gets the current mode of the selector input.
    /// This mode determines what happens and in what order when the external selector input is
    ///   used.
    /// 
    /// \param mode Variable to be filled with the selector mode
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getSelectorMode(uint8_t* mode);

    /// Sets the current mode of the selector input.
    /// This mode determines what happens and in what order when the external selector input is
    ///   used.
    /// 
    /// \param mode Mode to be set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setSelectorMode(const uint8_t mode);

    /// Gets the USB HighSpeed upstream port.
    /// 
    /// \param port The current upstream port.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getUpstreamHS(uint8_t* port);

    /// Sets the USB HighSpeed upstream port.
    /// 
    /// \param port The upstream port to set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setUpstreamHS(const uint8_t port);

    /// Gets the USB SuperSpeed upstream port.
    /// 
    /// \param port The current upstream port.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getUpstreamSS(uint8_t* port);

    /// Sets the USB SuperSpeed upstream port.
    /// 
    /// \param port The upstream port to set.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setUpstreamSS(const uint8_t port);

    /// Gets the current enabled overrides
    /// 
    /// \param overrides Bit mapped representation of the current override configuration.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getOverride(uint32_t* overrides);

    /// Sets the current enabled overrides
    /// 
    /// \param overrides Overrides to be set in a bit mapped representation.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setOverride(const uint32_t overrides);

    /// Sets the USB HighSpeed Max datarate
    /// 
    /// \param datarate Maximum datarate for the USB HighSpeed signals.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataHSMaxDatarate(const uint32_t datarate);

    /// Gets the USB HighSpeed Max datarate
    /// 
    /// \param datarate Current maximum datarate for the USB HighSpeed signals.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataHSMaxDatarate(uint32_t* datarate);

    /// Sets the USB SuperSpeed Max datarate
    /// 
    /// \param datarate Maximum datarate for the USB SuperSpeed signals.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr setDataSSMaxDatarate(const uint32_t datarate);

    /// Gets the USB SuperSpeed Max datarate
    /// 
    /// \param datarate Current maximum datarate for the USB SuperSpeed signals.
    /// 
    /// \return Returns \ref EntityReturnValues "common entity" return values
    aErr getDataSSMaxDatarate(uint32_t* datarate);



};

} // namespace BrainStem
} // namespace Acroname

#endif // defined(__cplusplus)

#endif // __AUTOGEN_USBSYSTEM_CPP_H__