/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: Entity_CCA.h
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2024 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////
#ifndef __Entity_CCA_H__
#define __Entity_CCA_H__

#include "CCA_Core.h"

#ifdef __cplusplus
extern "C" {
#endif

    /// \defgroup EntityReturnValues Common EntityClass Return Values
    /// Common EntityClass Return Values
    /// @{
    ///     - ::aErrNone - Action completed successfully.
    ///     - ::aErrTimeout - Request timed out without response.
    ///     - ::aErrConnection - No active link.
    /// @}

    /// A callUEI is a setUEI that has no data length.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    aLIBEXPORT void __stdcall entity_callUEI(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option);

    /// Set a byte value.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    /// \param inValue Byte value to be set
    aLIBEXPORT void __stdcall entity_setUEI8(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option, int inValue);

    /// Set a byte value with a subIndex.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR  or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    /// \param subIndex The sub index to be used.
    /// \param inValue Byte value to be set
    aLIBEXPORT void __stdcall entity_setUEI8SubIndex(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option, const unsigned char subIndex, int inValue);

    /// Get a byte value.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR and the requested value or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    aLIBEXPORT void __stdcall entity_getUEI8(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option);

    /// Get a byte value with a parameter.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR and the requested value or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    /// \param subIndex The sub index to be used.
    aLIBEXPORT void __stdcall entity_getUEI8SubIndex(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option, const unsigned char subIndex);

    /// Set a byte value.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    /// \param inValue 2 byte value to be set
    aLIBEXPORT void __stdcall entity_setUEI16(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option, const int inValue);

    /// Set a 2 byte value with a subIndex.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR  or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    /// \param subIndex The sub index to be used.
    /// \param inValue 2 byte value to be set
    aLIBEXPORT void __stdcall entity_setUEI16SubIndex(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option, const unsigned char subIndex, const int inValue);

    /// Get a 2 byte value.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR and the requested value or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    aLIBEXPORT void __stdcall entity_getUEI16(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option);

    /// Get a 2 byte value with a parameter.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR and the requested value or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    /// \param subIndex The sub index to be used.
    aLIBEXPORT void __stdcall entity_getUEI16SubIndex(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option, const unsigned char subIndex);

    /// Set a 4-byte value.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    /// \param inValue 4 byte value to be set
    aLIBEXPORT void __stdcall entity_setUEI32(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option, const int inValue);

    /// Set a 4 byte value with a subIndex.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR  or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    /// \param subIndex The sub index to be used.
    /// \param inValue 4 byte value to be set
    aLIBEXPORT void __stdcall entity_setUEI32SubIndex(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option, const unsigned char subIndex, const int inValue);

    /// Get a 4 byte value.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR and the requested value or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    aLIBEXPORT void __stdcall entity_getUEI32(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option);

    /// Get a 4 byte value with a parameter.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR and the requested value or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    /// \param subIndex The sub index to be used.
    aLIBEXPORT void __stdcall entity_getUEI32SubIndex(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option, const unsigned char subIndex);

    /// Set a multi-byte value.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    /// \param buffer Pointer to the start of the data to be sent
    /// \param bufferLength The amount of data to send in bytes.
    aLIBEXPORT void __stdcall entity_setUEIBytes ( unsigned int* id, struct Result* result, const unsigned char cmd , const unsigned char index, const unsigned char option, unsigned char * buffer, const int bufferLength ) ;

    /// Get a multi-byte value.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR and the requested value or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    /// \param buffer Pointer to a buffer to be filled by the call.
    /// \param bufferLength The size of the buffer in bytes.
    aLIBEXPORT void __stdcall entity_getUEIBytes ( unsigned int* id, struct Result* result, const unsigned char cmd , const unsigned char index, const unsigned char option, unsigned char * buffer, const int bufferLength ) ;

    /// Drain all packets matching this UEI from the packet fifo.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param option BrainStem Protocol option code.
    aLIBEXPORT void __stdcall entity_drainUEI(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const unsigned char option);

    /// Enables streaming for a specific command, option, and index.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param enable The enabled/disabled state to be applied.
    aLIBEXPORT void __stdcall entity_setStreamEnabled(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index, const bool enable);


    /// Gets all available stream values associated with the cmd and index of the called API.
    /// Keys can be decoded with Link::getStreamKeyElement.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR and the requested value or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    /// \param buffer Buffer of user allocated memory to be filled with stream data
    /// \param bufferLength Number of elements the buffer can hold.
    /// \details ::aErrParam if status or unloadedSize is null
    /// \details ::aErrResource - if the link is not valid
    /// \details ::aErrNone - success
    aLIBEXPORT void __stdcall entity_getStreamStatus(unsigned int* id,
                                                     struct Result* result,
                                                     const unsigned char cmd,
                                                     const unsigned char index,
                                                     struct StreamStatusEntry_CCA* buffer,
                                                     const unsigned int bufferLength);

    /// Resets the provided cmd and index combination to their default factory settings.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    aLIBEXPORT void __stdcall entity_resetEntityToFactoryDefaults(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index);

    /// Loads the provided command and index combination from memory.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    aLIBEXPORT void __stdcall entity_loadEntityFromStore(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index);

    /// Saves the provided command and index combination to memory.
    /// \param id Unique identifier for the internally created stem.
    /// \param result object, containing NO_ERROR or a non zero error code.
    /// \param cmd BrainStem Protocol command code
    /// \param index The index of the UEI
    aLIBEXPORT void __stdcall entity_saveEntityToStore(unsigned int* id, struct Result* result, const unsigned char cmd, const unsigned char index);
    
#ifdef __cplusplus
}
#endif

#endif
