/////////////////////////////////////////////////////////////////////
//                                                                 //
// file: aVersion_CCA.h
//                                                                 //
/////////////////////////////////////////////////////////////////////
//                                                                 //
// Copyright (c) 2024 Acroname Inc. - All Rights Reserved          //
//                                                                 //
// This file is part of the BrainStem release. See the license.txt //
// file included with this package or go to                        //
// https://acroname.com/software/brainstem-development-kit         //
// for full license details.                                       //
/////////////////////////////////////////////////////////////////////
#ifndef __version_CCA_H__
#define __version_CCA_H__

#include "CCA_Core.h"

/// \defgroup Version Versions
/// Functions for getting and comparing software and firmware version

#ifdef __cplusplus
extern "C" {
#endif

     /// Parses the major revision level from the given build number.
     /// \param result object, containing NO_ERROR or a non zero Error code.
     /// \param build The packed version number returned from the
     ///              system.getVersion call.
    aLIBEXPORT void __stdcall version_ParseMajor( struct Result* result, unsigned int build );

    /// Parses the minor revision level from the given build number.
    /// \param result object, containing NO_ERROR or a non zero Error code.
    /// \param build The packed version number returned from the
    ///              system.getVersion call.
    aLIBEXPORT void __stdcall version_ParseMinor( struct Result* result, unsigned int build );

    /// Parses the revision patch level from the given build number.
    /// \param result object, containing NO_ERROR or a non zero Error code.
    /// \param build The packed version number returned from the
    ///              system.getVersion call.
    aLIBEXPORT void __stdcall version_ParsePatch( struct Result* result, unsigned int build );

     /// Check if the given build version is of the legacy packing format
     /// \param result object, containing NO_ERROR or a non zero Error code.
     /// \param build The packed version number returned from the
     ///              system.getVersion call.
    aLIBEXPORT void __stdcall version_IsLegacyFormat( struct Result* result, unsigned int build );

    /// Return the major revision number for the software package.
    /// \param result Object containing aErrNone and the requested value on success.
    aLIBEXPORT void __stdcall version_GetMajor( struct Result* result );

    /// Return the minor revision number for the software package.
    /// \param result Object containing aErrNone and the requested value on success.
    aLIBEXPORT void __stdcall version_GetMinor( struct Result* result );

    /// Return the patch revision number for the software package.
    /// \param result Object containing aErrNone and the requested value on success.
    aLIBEXPORT void __stdcall version_GetPatch( struct Result* result );

    /// Check that the current software version is at least major.minor.patch
    /// \param result Object containing aErrNone and the requested value on success.
    /// \param major The major revision level.
    /// \param minor The minor revision level.
    /// \param patch The patch revision level.
    aLIBEXPORT void __stdcall version_IsAtLeast( struct Result* result, unsigned int major, unsigned int minor, unsigned int patch);

    /// Check that the supplied left hand side (lhs) version is at least (>=) the right hand side (rhs).
    /// \param result Object containing aErrNone and the requested value on success.
    /// \param major_lhs The lhs major revision level.
    /// \param minor_lhs The lhs minor revision level.
    /// \param patch_lhs The lhs patch revision level.
    /// \param major_rhs The rhs major revision level.
    /// \param minor_rhs The rhs minor revision level.
    /// \param patch_rhs The rhs patch revision level.
    aLIBEXPORT void __stdcall version_IsAtLeastCompare( struct Result* result, unsigned int major_lhs, unsigned int minor_lhs, unsigned int patch_lhs, unsigned int major_rhs, unsigned int minor_rhs, unsigned int patch_rhs);

    /// Packs the given version into a single integer
    /// \param result Object containing aErrNone and the requested value on success.
    /// \param major The major revision level.
    /// \param minor The minor revision level.
    /// \param patch The patch revision level.
    aLIBEXPORT void __stdcall version_Pack( struct Result* result, unsigned int major,  unsigned int minor, unsigned int patch);

    /// Provides a human readable string of the current software package
    /// \param result Object containing aErrNone and the requested value on success.
    /// \param buffer Pointer to a buffer to be filled.
    /// \param bufferLength Length of the provide buffer.
    aLIBEXPORT void __stdcall version_GetString(struct Result* result, unsigned char* buffer, unsigned int bufferLength);

#ifdef __cplusplus
}
#endif

#endif
